// SPDX-License-Identifier: LGPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* nymea-zigbee
* Zigbee integration module for nymea
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-zigbee.
*
* nymea-zigbee is free software: you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License
* as published by the Free Software Foundation, either version 3
* of the License, or (at your option) any later version.
*
* nymea-zigbee is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with nymea-zigbee. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef ZIGBEEBRIDGECONTROLLER_H
#define ZIGBEEBRIDGECONTROLLER_H

#include <QDir>
#include <QObject>
#include <QSerialPort>

#include "zigbee.h"

Q_DECLARE_METATYPE(QSerialPort::SerialPortError);

class ZigbeeBridgeController : public QObject
{
    Q_OBJECT
    friend class ZigbeeNetwork;

public:
    explicit ZigbeeBridgeController(QObject *parent = nullptr);
    ~ZigbeeBridgeController() = default;

    QString firmwareVersion() const;
    bool available() const;

    bool canUpdate() const;
    bool initiallyFlashed() const;
    bool updateRunning() const;

    // Optional update/initialize procedure for the zigbee controller
    virtual bool updateAvailable(const QString &currentVersion);
    virtual QString updateFirmwareVersion() const;
    virtual void startFirmwareUpdate();
    virtual void startFactoryResetUpdate();

protected:
    QString m_firmwareVersion;
    bool m_available = false;
    bool m_canUpdate = false;
    bool m_initiallyFlashed = false;
    bool m_updateRunning = false;
    QDir m_settingsDirectory = QDir("/etc/nymea/");

    void setAvailable(bool available);
    void setFirmwareVersion(const QString &firmwareVersion);

    virtual void initializeUpdateProvider();

private:
    void setSettingsDirectory(const QDir &settingsDirectory);

signals:
    void availableChanged(bool available);
    void firmwareVersionChanged(const QString &firmwareVersion);
    void canUpdateChanged(bool canUpdate);
    void updateRunningChanged(bool updateRunning);

    // APS notifications
    void apsDataConfirmReceived(const Zigbee::ApsdeDataConfirm &confirm);
    void apsDataIndicationReceived(const Zigbee::ApsdeDataIndication &indication);
    void apsDataAckReceived(const Zigbee::ApsdeDataAck &acknowledgement);

};

#endif // ZIGBEEBRIDGECONTROLLER_H
