// SPDX-License-Identifier: LGPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* nymea-zigbee
* Zigbee integration module for nymea
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-zigbee.
*
* nymea-zigbee is free software: you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License
* as published by the Free Software Foundation, either version 3
* of the License, or (at your option) any later version.
*
* nymea-zigbee is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with nymea-zigbee. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef ZIGBEECLUSTERONOFF_H
#define ZIGBEECLUSTERONOFF_H

#include <QObject>

#include "zcl/zigbeecluster.h"
#include "zcl/zigbeeclusterreply.h"

class ZigbeeNode;
class ZigbeeNetwork;
class ZigbeeNodeEndpoint;
class ZigbeeNetworkReply;

class ZigbeeClusterOnOff : public ZigbeeCluster
{
    Q_OBJECT

    friend class ZigbeeNode;
    friend class ZigbeeNetwork;

public:
    enum Attribute {
        AttributeOnOff = 0x0000,
        AttributeGlobalSceneControl = 0x4000,
        AttributeOnTime = 0x4001,
        AttributeOffTime = 0x4002
    };
    Q_ENUM(Attribute)

    enum Command {
        CommandOff = 0x00,
        CommandOn = 0x01,
        CommandToggle = 0x02,
        CommandOffWithEffect = 0x40,
        CommandOnWithRecallGlobalScene = 0x41,
        CommandOnWithTimedOff = 0x42
    };
    Q_ENUM(Command)

    enum Effect {
        EffectDelayedAllOff = 0x00,
        EffectDyingLight = 0x01
    };
    Q_ENUM(Effect)

    explicit ZigbeeClusterOnOff(ZigbeeNetwork *network, ZigbeeNode *node, ZigbeeNodeEndpoint *endpoint, Direction direction, QObject *parent = nullptr);

    ZigbeeClusterReply *commandOff();
    ZigbeeClusterReply *commandOn();
    ZigbeeClusterReply *commandToggle();
    ZigbeeClusterReply *commandOffWithEffect(Effect effect, quint8 effectVariant = 0x00);
    ZigbeeClusterReply *commandOnWithRecallGlobalScene();
    ZigbeeClusterReply *commandOnWithTimedOff(bool acceptOnlyWhenOn, quint16 onTime, quint16 offWaitTime);

    bool power() const;

private:
    bool m_power = false;

    void setAttribute(const ZigbeeClusterAttribute &attribute) override;

protected:
    void processDataIndication(ZigbeeClusterLibrary::Frame frame) override;

signals:
    // Server cluster signals
    void powerChanged(bool power);

    // Client cluster signals
    void commandReceived(ZigbeeClusterOnOff::Command command, const QByteArray &parameters, quint8 transactionSequenceNumber);
    // On and off time is in 1/10 seconds
    void commandOnWithTimedOffReceived(bool acceptOnlyWhenOn, quint16 onTime, quint16 offTime, quint8 transactionSequenceNumber);

    // On and off time is in 1/10 seconds
    void commandOffWithEffectReceived(ZigbeeClusterOnOff::Effect effect, quint8 effectVariant, quint8 transactionSequenceNumber);
};

#endif // ZIGBEECLUSTERONOFF_H
