// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-nymea.
*
* nymea-energy-plugin-nymea.s free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-nymea.s distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-nymea. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "energymanagerdbusinterface.h"

#include <QDBusConnection>
#include <QLoggingCategory>

#include <QLoggingCategory>
Q_DECLARE_LOGGING_CATEGORY(dcNymeaEnergy)

#include "../smartchargingmanager.h"

static const QString kDbusService = QStringLiteral("io.nymea.energymanager");
static const QString kDbusPath = QStringLiteral("/io/nymea/energymanager");

EnergyManagerDbusInterface::EnergyManagerDbusInterface(SmartChargingManager *smartChargingManager, QObject *parent):
    QObject(parent),
    m_smartChargingManager(smartChargingManager)
{
    QDBusConnection connection = QDBusConnection::systemBus();

    if (!connection.isConnected()) {
        qCWarning(dcNymeaEnergy()) << "Could not connect to DBus system bus. Charging info DBus interface not available.";
        return;
    }

    if (!connection.registerService(kDbusService)) {
        qCWarning(dcNymeaEnergy()) << "Failed to register DBus service" << kDbusService << "Charging info DBus interface not available.";
        return;
    }

    if (!connection.registerObject(kDbusPath, this, QDBusConnection::ExportAllSlots | QDBusConnection::ExportAllSignals)) {
        qCWarning(dcNymeaEnergy()) << "Failed to register DBus object path" << kDbusPath;
        return;
    }

    connect(m_smartChargingManager, &SmartChargingManager::chargingInfoAdded, this, [this](const ChargingInfo &info){
        emit chargingInfoAdded(serializeChargingInfo(info));
    });
    connect(m_smartChargingManager, &SmartChargingManager::chargingInfoRemoved, this, [this](const ThingId &evChargerId){
        emit chargingInfoRemoved(evChargerId.toString());
    });
    connect(m_smartChargingManager, &SmartChargingManager::chargingInfoChanged, this, [this](const ChargingInfo &info){
        emit chargingInfoChanged(serializeChargingInfo(info));
    });
}

QVariantList EnergyManagerDbusInterface::chargingInfos() const
{
    QVariantList infos;
    if (!m_smartChargingManager) {
        return infos;
    }

    foreach (const ChargingInfo &info, m_smartChargingManager->chargingInfos()) {
        infos.append(serializeChargingInfo(info));
    }

    return infos;
}

QVariantMap EnergyManagerDbusInterface::serializeChargingInfo(const ChargingInfo &info) const
{
    QVariantMap map;
    map.insert(QStringLiteral("evChargerId"), info.evChargerId().toString());

    if (!info.assignedCarId().isNull()) {
        map.insert(QStringLiteral("assignedCarId"), info.assignedCarId().toString());
    }

    map.insert(QStringLiteral("chargingMode"), static_cast<int>(info.chargingMode()));
    map.insert(QStringLiteral("chargingState"), static_cast<int>(info.chargingState()));

    if (info.endDateTime().isValid()) {
        map.insert(QStringLiteral("endDateTime"), info.endDateTime().toString(Qt::ISODate));
    }

    if (!info.repeatDays().isEmpty()) {
        map.insert(QStringLiteral("repeatDays"), QVariant::fromValue(info.repeatDays()));
    }

    map.insert(QStringLiteral("targetPercentage"), info.targetPercentage());
    map.insert(QStringLiteral("spotMarketChargingEnabled"), info.spotMarketChargingEnabled());
    map.insert(QStringLiteral("dailySpotMarketPercentage"), info.dailySpotMarketPercentage());

    return map;
}
