// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-nymea.
*
* nymea-energy-plugin-nymea.s free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-nymea.s distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-nymea. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "energyexperienceenergymock.h"

#include "energyjsonhandler.h"
#include "energymanagermock.h"
#include "energyplugin.h"

#include "../../../energyplugin/smartchargingmanager.h"
#include "../../../energyplugin/nymeaenergyjsonhandler.h"
#include "../../../energyplugin/energymanagerconfiguration.h"
#include "../../../energyplugin/spotmarket/spotmarketmanager.h"

#include <jsonrpc/jsonrpcserver.h>
#include <loggingcategories.h>

#include <QDir>
#include <QCoreApplication>
#include <QPluginLoader>

NYMEA_LOGGING_CATEGORY(dcEnergyExperience, "EnergyExperience")

ExperiencePluginEnergyMock::ExperiencePluginEnergyMock()
{

}

void ExperiencePluginEnergyMock::setEnergyLogDbFilePath(const QString &energyLogDbFilePath)
{
    m_energyLogDbFilePath = energyLogDbFilePath;
}

void ExperiencePluginEnergyMock::init()
{
    qCDebug(dcEnergyExperience()) << "Initializing mock energy experience using DB" << m_energyLogDbFilePath;
    QFile logdbFile(m_energyLogDbFilePath);
    if (!logdbFile.open(QIODevice::ReadOnly)) {
        qCWarning(dcEnergyExperience()) << "Failed to load energy log db" << logdbFile.errorString();
        return;
    }

    // Create a temporary copy since read only did not work as expeected
    QString finalDbName = "/tmp/energylogs.sqlite";
    if (QFile::exists(finalDbName))
        QFile::remove(finalDbName);

    if (!logdbFile.copy(finalDbName)) {
        qCWarning(dcEnergyExperience()) << "Failed to copy energy log db" << m_energyLogDbFilePath << "->" << finalDbName << logdbFile.errorString();
        return;
    }



    m_energyManager = new EnergyManagerMock(thingManager(), finalDbName, this);
    m_energyJsonHandler = new EnergyJsonHandler(m_energyManager, this);
    jsonRpcServer()->registerExperienceHandler(m_energyJsonHandler, 0, 2);

    // This is the init call of our nymea.energy plugin
    EnergyManagerConfiguration *configuration = new EnergyManagerConfiguration(this);
    QNetworkAccessManager *networkManager = new QNetworkAccessManager(this);
    m_spotMarketManager = new SpotMarketManager(networkManager, this);
    m_smartChargingManager = new SmartChargingManager(m_energyManager, thingManager(), m_spotMarketManager, configuration, this);

    m_nymeaEnergyJsonHandler = new NymeaEnergyJsonHandler(m_spotMarketManager, m_smartChargingManager, this);
    jsonRpcServer()->registerExperienceHandler(m_nymeaEnergyJsonHandler, 0, 2);

}

EnergyManagerMock *ExperiencePluginEnergyMock::energyManager() const
{
        return m_energyManager;
}

SmartChargingManager *ExperiencePluginEnergyMock::smartChargingManager() const
{
    return m_smartChargingManager;
}

SpotMarketManager *ExperiencePluginEnergyMock::spotMarketManager() const
{
    return m_spotMarketManager;
}

void ExperiencePluginEnergyMock::testCall()
{
    qCDebug(dcEnergyExperience()) << "Test call!";
}
