// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-nymea.
*
* nymea-energy-plugin-nymea.s free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-nymea.s distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-nymea. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef SPOTMARKETDATAPROVIDER_H
#define SPOTMARKETDATAPROVIDER_H

#include <QUrl>
#include <QUuid>
#include <QObject>
#include <QDebug>
#include <QLocale>
#include <QNetworkAccessManager>

#include "../types/scoreentry.h"

class SpotMarketProviderInfo
{
    Q_GADGET
    Q_PROPERTY(QUuid providerId READ providerId CONSTANT)
    Q_PROPERTY(QString name READ name CONSTANT)
    Q_PROPERTY(uint country READ country CONSTANT)
    Q_PROPERTY(QUrl website READ website CONSTANT)

public:
    SpotMarketProviderInfo() = default;
    SpotMarketProviderInfo(const QUuid &providerId, const QString &name, QLocale::Country country, const QUrl &website) :
        m_providerId{providerId},
        m_name{name},
        m_country{country},
        m_website{website} {
    }

    QUuid providerId() const {
        return m_providerId;
    }

    QString name() const {
        return m_name;
    }

    QLocale::Country country() const {
        return m_country;
    }

    QUrl website() const {
        return m_website;
    }

private:
    QUuid m_providerId;
    QString m_name;
    QLocale::Country m_country = QLocale::Austria;
    QUrl m_website;

};

//typedef QList<SpotMarketProviderInfo> SpotMarketProviderInfos;

class SpotMarketProviderInfos: public QList<SpotMarketProviderInfo>
{
    Q_GADGET
    Q_PROPERTY(int count READ count)
public:
    SpotMarketProviderInfos() = default;
    SpotMarketProviderInfos(const QList<SpotMarketProviderInfo> &other) : QList<SpotMarketProviderInfo>(other) { }
    SpotMarketProviderInfos(std::initializer_list<SpotMarketProviderInfo> args):QList(args) { }
    Q_INVOKABLE QVariant get(int index) const {
        return QVariant::fromValue(at(index));
    }
    Q_INVOKABLE void put(const QVariant &variant) {
        append(variant.value<SpotMarketProviderInfo>());
    }
};
Q_DECLARE_METATYPE(SpotMarketProviderInfos)


class SpotMarketDataProvider : public QObject
{
    Q_OBJECT
public:
    explicit SpotMarketDataProvider(QNetworkAccessManager *networkManager, QObject *parent = nullptr);
    virtual ~SpotMarketDataProvider() = default;

    virtual QUuid providerId() const = 0;

    SpotMarketProviderInfo info() const;

    bool enabled() const;
    bool available() const;

    ScoreEntries scoreEntries() const;

public slots:
    virtual void enable() = 0;
    virtual void disable() = 0;
    virtual void refreshData() = 0;

signals:
    void enabledChanged(bool enabled);
    void availableChanged(bool available);
    void scoreEntriesChanged(const ScoreEntries &scoreEntries);

protected:
    QNetworkAccessManager *m_networkManager = nullptr;
    SpotMarketProviderInfo m_info;
    bool m_enabled = false;
    bool m_available = false;
    QString m_cacheFilePath;

    ScoreEntries m_scoreEntries;

    void cacheDataEntries(const ScoreEntries &scoreEntries);
    ScoreEntries loadCachedDataEntries();
    bool clearCache();
};

QDebug operator<<(QDebug debug, SpotMarketDataProvider *provider);
QDebug operator<<(QDebug debug, const SpotMarketProviderInfo &providerInfo);


#endif // SPOTMARKETDATAPROVIDER_H
