/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright 2013 - 2022, nymea GmbH
* Contact: contact@nymea.io
*
* This file is part of nymea.
* This project including source code and documentation is protected by
* copyright law, and remains the property of nymea GmbH. All rights, including
* reproduction, publication, editing and translation, are reserved. The use of
* this project is subject to the terms of a license agreement to be concluded
* with nymea GmbH in accordance with the terms of use of nymea GmbH, available
* under https://nymea.io/license
*
* GNU General Public License Usage
* Alternatively, this project may be redistributed and/or modified under the
* terms of the GNU General Public License as published by the Free Software
* Foundation, GNU version 3. This project is distributed in the hope that it
* will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
* of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
* Public License for more details.
*
* You should have received a copy of the GNU General Public License along with
* this project. If not, see <https://www.gnu.org/licenses/>.
*
* For any further details and any questions please contact us under
* contact@nymea.io or see our FAQ/Licensing Information on
* https://nymea.io/license/faq
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef ENERGYTESTBASE_H
#define ENERGYTESTBASE_H

#include <QObject>
#include <QtTest>
#include <QDateTime>
#include <QTcpSocket>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QLoggingCategory>

#include <nymeatestbase.h>

#include <nymeacore.h>
#include <nymeasettings.h>
#include <servers/mocktcpserver.h>

#include <energymanager.h>

static QUuid mockMeterThingClassId = QUuid("2721a051-6e12-471a-baba-21d87c4cebc9");
static QUuid mockChargerThingClassId = QUuid("5a3ae99f-c7da-46df-9104-f477be4606b7");
static QUuid mockChargerWithPhaseSwitchingThingClassId = QUuid("9208d9f0-280c-469d-a145-106f3277470c");
static QUuid mockSimpleChargerThingClassId = QUuid("29bcf255-b654-4764-be92-399bc26fe7c3");
static QUuid mockCarThingClassId = QUuid("4513f801-836e-40a7-8784-c02650a9bdc6");
static QUuid mockEnergyStorageThingClassId = QUuid("d0d5bbf0-249c-46ed-ac6a-5f271b2b0b0f");

using namespace nymeaserver;

class EnergyTestBase : public NymeaTestBase
{
    Q_OBJECT
public:
    explicit EnergyTestBase(QObject *parent = nullptr);

    inline void verifyThingError(const QVariant &response, Thing::ThingError error = Thing::ThingErrorNoError) {
        verifyError(response, "thingError", enumValueName(error));
    }

    inline void verifyEnergyError(const QVariant &response, EnergyManager::EnergyError error = EnergyManager::EnergyErrorNoError) {
        verifyError(response, "energyError", enumValueName(error));
    }

    static QDateTime utcDateTime(const QDate &date, const QTime &time);

protected slots:
    void initTestCase(const QString &loggingRules = QString(), bool checkExperienceLoaded = true, bool disableLogEngine = true);
    void cleanupTestCase();

    void init();
    void cleanup();

public:
    QNetworkReply *setMeterStates(const QVariantMap &phasesPower, bool connected = true, quint16 port = 26655);
    QNetworkReply *setCarStates(uint batteryLevel, uint capacity, uint minChargingCurrent, uint phaseCount, quint16 port = 26656);
    QNetworkReply *setChargerStates(bool connected, bool power, bool pluggedIn, const QString &phases, int maxChargingCurrent, int maxChargingCurrentMaxValue, quint16 port = 26657);
    QNetworkReply *setChargerWithPhaseCountSwitchingStates(bool connected, bool power, bool pluggedIn, const QString &phases, int maxChargingCurrent, int maxChargingCurrentMaxValue, uint desiredPhaseCount, quint16 port = 26658);
    QNetworkReply *setSimpleChargerStates(bool connected, bool power, bool pluggedIn, int phaseCount, int maxChargingCurrent, quint16 port = 26659);
    QNetworkReply *setEnergyStorageStates(uint batteryLevel, int currentPower, quint16 port = 26660);

    QNetworkReply *getActionHistory(quint16 port);
    QNetworkReply *clearActionHistroy(quint16 port);

    QUuid addMeter(quint16 port = 26655);
    QUuid addCar(quint16 port = 26656);
    QUuid addCharger(const QString &phases = "All", double maxChargingCurrentUpperLimit = 32, quint16 port = 26657);
    QUuid addChargerWithPhaseCountSwitching(const QString &phases = "All", double maxChargingCurrentUpperLimit = 32, quint16 port = 26658);
    QUuid addSimpleCharger(double maxChargingCurrentUpperLimit = 32, quint16 port = 26659);
    QUuid addEnergyStorage(uint capacity = 10, double maxChargingPowerUpperLimit = 5000, double maxDischargingPowerUpperLimit = 11500, quint16 port = 26660);

    void removeDevices();
    QVariant removeDevice(const QUuid &thingId);

protected:
    QNetworkAccessManager *m_networkAccessManager = nullptr;

    quint16 m_mockMeterDefaultPort = 26655;
    quint16 m_mockCarDefaultPort = 26656;
    quint16 m_mockChargerDefaultPort = 26657;
    quint16 m_mockChargerWithPhaseCountSwitchingDefaultPort = 26658;
    quint16 m_mockSimpleChargerDefaultPort = 26659;
    quint16 m_mockEnergyStorageDefaultPort = 26660;

    bool verifyActionExecuted(const QVariantList &actionHistory, const QString &actionName);
    QVariant getLastValueFromExecutedAction(const QVariantList &actionHistory, const QString &actionName, const QString &paramName);

};

#endif // ENERGYTESTBASE_H
