/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright 2013 - 2022, nymea GmbH
* Contact: contact@nymea.io
*
* This file is part of nymea.
* This project including source code and documentation is protected by
* copyright law, and remains the property of nymea GmbH. All rights, including
* reproduction, publication, editing and translation, are reserved. The use of
* this project is subject to the terms of a license agreement to be concluded
* with nymea GmbH in accordance with the terms of use of nymea GmbH, available
* under https://nymea.io/license
*
* GNU Lesser General Public License Usage
* Alternatively, this project may be redistributed and/or modified under the
* terms of the GNU Lesser General Public License as published by the Free
* Software Foundation; version 3. This project is distributed in the hope that
* it will be useful, but WITHOUT ANY WARRANTY; without even the implied
* warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with this project. If not, see <https://www.gnu.org/licenses/>.
*
* For any further details and any questions please contact us under
* contact@nymea.io or see our FAQ/Licensing Information on
* https://nymea.io/license/faq
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef SCOREENTRY_H
#define SCOREENTRY_H

#include <QDebug>
#include <QDateTime>
#include <QMetaObject>
#include <qhashfunctions.h>

#include "timeframe.h"

#if QT_VERSION < QT_VERSION_CHECK(6, 0, 0)
using qhash_result_t = uint;
#else
using qhash_result_t = size_t;
#endif

class ScoreEntry : public TimeFrame
{
    Q_GADGET
    Q_PROPERTY(QDateTime startDateTime READ startDateTime WRITE setStartDateTime)
    Q_PROPERTY(QDateTime endDateTime READ endDateTime WRITE setEndDateTime)
    Q_PROPERTY(float value READ value WRITE setValue)
    Q_PROPERTY(float weighting READ weighting WRITE setWeighting)

public:
    ScoreEntry();

    // Required for recalculating weighting on new data
    double value() const;
    void setValue(double value);

    // How good is this entry compaired to the others [0, 1]
    // 0 - The worst entry
    // 1 - The best entry
    float weighting() const;
    void setWeighting(float weighting);

    bool isNull() const;

    bool operator==(const ScoreEntry &other) const;
    bool operator!=(const ScoreEntry &other) const;

private:
    double m_value = 0;
    float m_weighting = 0;

};

qhash_result_t qHash(const ScoreEntry &scoreEntry, qhash_result_t seed = 0);

Q_DECLARE_METATYPE(ScoreEntry)

class ScoreEntries: public QList<ScoreEntry>
{
    Q_GADGET
    Q_PROPERTY(int count READ count)
public:
    ScoreEntries() = default;
    ScoreEntries(const QList<ScoreEntry> &other);
    Q_INVOKABLE QVariant get(int index) const;
    Q_INVOKABLE void put(const QVariant &variant);

    int availableFutureSchedules(const QDateTime &currentDateTime) const;

    // index 0 has the best weighting (1)
    void sortByWeighting();

    // index 0 is the oldest entry
    void sortByStartDateTime();

    ScoreEntry getScoreEntry(const QDateTime &currentDateTime = QDateTime::currentDateTime()) const;

    bool availableUpTo(const QDateTime &dateTime) const;
};

Q_DECLARE_METATYPE(ScoreEntries)

QDebug operator<<(QDebug debug, const ScoreEntry &dataEntry);
QDebug operator<<(QDebug debug, const ScoreEntries &scoreEntries);

#endif // SCOREENTRY_H
