// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-chargingsessions.
*
* nymea-energy-plugin-chargingsessions is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-chargingsessions is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-chargingsessions. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "testchargingsessionsdatabase.h"
#include "../../../energyplugin/chargingsessionsdatabase.h"
#include "../../../energyplugin/chargingsessionssettings.h"
#include "../../../energyplugin/chargingsessionsmanager.h"

#include <QSignalSpy>
#include <QDateTime>
#include <QTemporaryDir>
#include <QDir>

ChargingSessionsDatabaseTests::ChargingSessionsDatabaseTests(QObject *parent)
    : QObject(parent)
{
    // Important for settings
    QCoreApplication::instance()->setOrganizationName("nymea-test");
}

void ChargingSessionsDatabaseTests::initTestCase()
{
    QFile file(ChargingSessionsSettings::databaseName());
    if (file.exists())
        file.remove();
}

void ChargingSessionsDatabaseTests::cleanupTestCase()
{
    ChargingSessionsDatabase database(ChargingSessionsSettings::databaseName());
    database.wipeDatabase();
}

void ChargingSessionsDatabaseTests::databaseLogStartSession_data()
{
    QTest::addColumn<QUuid>("evChargerId");
    QTest::addColumn<QString>("evChargerName");
    QTest::addColumn<QString>("serialNumber");
    QTest::addColumn<QUuid>("carId");
    QTest::addColumn<QString>("carName");
    QTest::addColumn<QDateTime>("startDateTime");
    QTest::addColumn<double>("energyStart");
    QTest::addColumn<bool>("wipeDatabase");
    QTest::addColumn<int>("expectedSessionId");

    // NOTE: always wipe the DB in the last test row!

    QTest::newRow("Default") << QUuid::createUuid() << "Super cool EV carger" << "666" << QUuid::createUuid() << "Tuuut tuuut" << QDateTime::currentDateTime() << 1000.0 << false << 1;
    QTest::newRow("Default") << QUuid::createUuid() << "Super cool EV carger 2" << "667" << QUuid::createUuid() << "Tuuut tuuut" << QDateTime::currentDateTime() << 1000.0 << true << 2;
}

void ChargingSessionsDatabaseTests::databaseLogStartSession()
{
    QFETCH(QUuid, evChargerId);
    QFETCH(QString, evChargerName);
    QFETCH(QString, serialNumber);
    QFETCH(QUuid, carId);
    QFETCH(QString, carName);
    QFETCH(QDateTime, startDateTime);
    QFETCH(double, energyStart);
    QFETCH(bool, wipeDatabase);
    QFETCH(int, expectedSessionId);

    ChargingSessionsDatabase database(ChargingSessionsSettings::databaseName());
    QCOMPARE(database.databaseName(), ChargingSessionsSettings::databaseName());
    QSignalSpy signalSpy(&database, &ChargingSessionsDatabase::databaseSessionAdded);
    database.logStartSession(evChargerId, evChargerName, serialNumber, carId, carName, startDateTime, energyStart);
    if (signalSpy.count() == 0) {
        signalSpy.wait();
    }

    QCOMPARE(signalSpy.count(), 1);

    ThingId receivedChargerId = qvariant_cast<ThingId>(signalSpy.at(0).at(0));
    int receivedSessionId = signalSpy.at(0).at(1).toInt();

    QCOMPARE(receivedSessionId, expectedSessionId);
    QCOMPARE(receivedChargerId, evChargerId);

    if(wipeDatabase)
        database.wipeDatabase();
}


void ChargingSessionsDatabaseTests::fullSessionLogTotalEnergyConsumed()
{
    QUuid evChargerId = QUuid::createUuid();
    QString evChargerName = "Charge Norris";
    QUuid carId = QUuid::createUuid();
    QString serialNumber = "NEEDC0FFE";
    int sessionId = -1;
    QString carName = "Tüüüt tüüüüüt";
    QDateTime startDateTime = QDateTime::currentDateTime();

    double totalEnergyConsumed = 1000.0;
    double startEnergy = totalEnergyConsumed;
    double secondsPassed = 0;

    ChargingSessionsDatabase database(ChargingSessionsSettings::databaseName());
    QCOMPARE(database.databaseName(), ChargingSessionsSettings::databaseName());

    // Start session (car gets plugged in)
    QSignalSpy signalAddEntrySpy(&database, &ChargingSessionsDatabase::databaseSessionAdded);
    database.logStartSession(evChargerId, evChargerName, serialNumber, carId, carName, startDateTime, totalEnergyConsumed);
    if (signalAddEntrySpy.count() == 0) {
        signalAddEntrySpy.wait();
    }

    QCOMPARE(signalAddEntrySpy.count(), 1);
    ThingId receivedChargerId = qvariant_cast<ThingId>(signalAddEntrySpy.at(0).at(0));
    int receivedSessionId = signalAddEntrySpy.at(0).at(1).toInt();

    QCOMPARE(receivedSessionId, 1);
    sessionId = receivedSessionId;
    QCOMPARE(receivedChargerId, evChargerId);

    // Consume energy...
    totalEnergyConsumed += 2.5;
    secondsPassed += 3600;

    QSignalSpy updateEnergySignalSpy(&database, &ChargingSessionsDatabase::databaseSessionUpdated);
    database.updateTotalEnergyConsumed(sessionId, totalEnergyConsumed, startDateTime.addSecs(secondsPassed));
    if (updateEnergySignalSpy.count() == 0) {
        updateEnergySignalSpy.wait();
    }

    QCOMPARE(updateEnergySignalSpy.count(), 1);

    FetchDataReply *fetchReply = database.fetchCarSessions(carId);
    QSignalSpy fetchReplySignalSpy(fetchReply, &FetchDataReply::finished);
    if (fetchReplySignalSpy.count() == 0) {
        fetchReplySignalSpy.wait();
    }

    QCOMPARE(fetchReplySignalSpy.count(), 1);
    QCOMPARE(fetchReply->error(), ChargingSessionsManager::ChargingSessionsErrorNoError);
    QVERIFY(fetchReply->sessions().isEmpty());

    secondsPassed += 300; // 5 min later

    // End session (car gets plugged out)
    QSignalSpy signalUpdateEnergySpy(&database, &ChargingSessionsDatabase::databaseSessionUpdated);
    database.logEndSession(sessionId, carId, carName, startDateTime.addSecs(secondsPassed), totalEnergyConsumed);
    if (signalUpdateEnergySpy.count() == 0) {
        signalUpdateEnergySpy.wait();
    }

    QCOMPARE(signalUpdateEnergySpy.count(), 1);
    receivedSessionId = signalAddEntrySpy.at(0).at(1).toInt();
    QCOMPARE(receivedSessionId, 1);

    FetchDataReply *fetchReply2 = database.fetchCarSessions(carId);
    QSignalSpy fetchReply2SignalSpy(fetchReply2, &FetchDataReply::finished);
    if (fetchReply2SignalSpy.count() == 0) {
        fetchReply2SignalSpy.wait();
    }

    QCOMPARE(fetchReply2SignalSpy.count(), 1);
    QCOMPARE(fetchReply2->error(), ChargingSessionsManager::ChargingSessionsErrorNoError);

    Session session = fetchReply2->sessions().first();
    QCOMPARE(session.sessionId().toInt(), sessionId);
    QCOMPARE(session.chargerName(), evChargerName);
    QCOMPARE(session.carName(), carName);
    QCOMPARE(session.chargerSerialNumber(), serialNumber);
    QCOMPARE(session.startTimestamp().toSecsSinceEpoch(), startDateTime.toSecsSinceEpoch());
    QCOMPARE(session.endTimestamp().toSecsSinceEpoch(), startDateTime.addSecs(secondsPassed).toSecsSinceEpoch());
    QCOMPARE(session.sessionEnergy(), totalEnergyConsumed - startEnergy);
    QCOMPARE(session.energyStart(), startEnergy);
    QCOMPARE(session.energyEnd(), totalEnergyConsumed);

    database.wipeDatabase();
}

void ChargingSessionsDatabaseTests::fullSessionLogSessionEnergy()
{
    QUuid evChargerId = QUuid::createUuid();
    QString evChargerName = "Charge Norris";
    QUuid carId = QUuid::createUuid();
    QString serialNumber = "NEEDC0FFE";
    int sessionId = -1;
    QString carName = "Tüüüt tüüüüüt";
    QDateTime startDateTime = QDateTime::currentDateTime();

    double sessionEnergy = 0;
    double secondsPassed = 0;

    ChargingSessionsDatabase database(ChargingSessionsSettings::databaseName());
    QCOMPARE(database.databaseName(), ChargingSessionsSettings::databaseName());

    // Start session (car gets plugged in)
    QSignalSpy signalAddEntrySpy(&database, &ChargingSessionsDatabase::databaseSessionAdded);
    database.logStartSession(evChargerId, evChargerName, serialNumber, carId, carName, startDateTime);
    if (signalAddEntrySpy.count() == 0) {
        signalAddEntrySpy.wait();
    }

    QCOMPARE(signalAddEntrySpy.count(), 1);
    ThingId receivedChargerId = qvariant_cast<ThingId>(signalAddEntrySpy.at(0).at(0));
    int receivedSessionId = signalAddEntrySpy.at(0).at(1).toInt();

    QCOMPARE(receivedSessionId, 1);
    sessionId = receivedSessionId;
    QCOMPARE(receivedChargerId, evChargerId);

    // Consume energy...
    sessionEnergy += 2.5;
    secondsPassed += 3600;

    QSignalSpy updateEnergySignalSpy(&database, &ChargingSessionsDatabase::databaseSessionUpdated);
    database.updateSessionEnergy(sessionId, sessionEnergy, startDateTime.addSecs(secondsPassed));
    if (updateEnergySignalSpy.count() == 0) {
        updateEnergySignalSpy.wait();
    }

    QCOMPARE(updateEnergySignalSpy.count(), 1);

    updateEnergySignalSpy.clear();

    // Consume energy...
    sessionEnergy += 2.5;
    secondsPassed += 3600;

    database.updateSessionEnergy(sessionId, sessionEnergy, startDateTime.addSecs(secondsPassed));
    if (updateEnergySignalSpy.count() == 0) {
        updateEnergySignalSpy.wait();
    }

    QCOMPARE(updateEnergySignalSpy.count(), 1);

    FetchDataReply *fetchReply = database.fetchCarSessions(carId);
    QSignalSpy fetchReplySignalSpy(fetchReply, &FetchDataReply::finished);
    if (fetchReplySignalSpy.count() == 0) {
        fetchReplySignalSpy.wait();
    }

    QCOMPARE(fetchReplySignalSpy.count(), 1);
    QCOMPARE(fetchReply->error(), ChargingSessionsManager::ChargingSessionsErrorNoError);
    QVERIFY(fetchReply->sessions().isEmpty());

    secondsPassed += 300; // 5 min later

    updateEnergySignalSpy.clear();

    // End session (car gets plugged out)
    QSignalSpy signalUpdateEnergySpy(&database, &ChargingSessionsDatabase::databaseSessionUpdated);
    database.logEndSession(sessionId, carId, carName, startDateTime.addSecs(secondsPassed));
    if (signalUpdateEnergySpy.count() == 0) {
        signalUpdateEnergySpy.wait();
    }

    QCOMPARE(signalUpdateEnergySpy.count(), 1);
    receivedSessionId = signalAddEntrySpy.at(0).at(1).toInt();
    QCOMPARE(receivedSessionId, 1);

    FetchDataReply *fetchReply2 = database.fetchCarSessions(carId);
    QSignalSpy fetchReply2SignalSpy(fetchReply2, &FetchDataReply::finished);
    if (fetchReply2SignalSpy.count() == 0) {
        fetchReply2SignalSpy.wait();
    }

    QCOMPARE(fetchReply2SignalSpy.count(), 1);
    QCOMPARE(fetchReply2->error(), ChargingSessionsManager::ChargingSessionsErrorNoError);

    Session session = fetchReply2->sessions().first();
    QCOMPARE(session.sessionId().toInt(), sessionId);
    QCOMPARE(session.chargerName(), evChargerName);
    QCOMPARE(session.carName(), carName);
    QCOMPARE(session.chargerSerialNumber(), serialNumber);
    QCOMPARE(session.startTimestamp().toSecsSinceEpoch(), startDateTime.toSecsSinceEpoch());
    QCOMPARE(session.endTimestamp().toSecsSinceEpoch(), startDateTime.addSecs(secondsPassed).toSecsSinceEpoch());
    QCOMPARE(session.sessionEnergy(), sessionEnergy);
    QCOMPARE(session.energyStart(), 0);
    QCOMPARE(session.energyEnd(), 0);
}

void ChargingSessionsDatabaseTests::fetchCarSessionsTimeRange()
{
    QTemporaryDir tempDir;
    QVERIFY(tempDir.isValid());
    const QString databaseName = tempDir.path() + QDir::separator() + QStringLiteral("chargingsessions.db");

    QUuid evChargerId = QUuid::createUuid();
    QString evChargerName = QStringLiteral("Charger");
    QString serialNumber = QStringLiteral("SERIAL");
    QUuid carId = QUuid::createUuid();
    QString carName = QStringLiteral("Car");

    const QDateTime baseTime = QDateTime::currentDateTime().addDays(-1);

    ChargingSessionsDatabase database(databaseName);

    // Session 1: baseTime .. baseTime+60
    QSignalSpy sessionAddedSpy1(&database, &ChargingSessionsDatabase::databaseSessionAdded);
    database.logStartSession(evChargerId, evChargerName, serialNumber, carId, carName, baseTime);
    if (sessionAddedSpy1.count() == 0) {
        sessionAddedSpy1.wait();
    }
    QCOMPARE(sessionAddedSpy1.count(), 1);
    const int sessionId1 = sessionAddedSpy1.at(0).at(1).toInt();

    QSignalSpy sessionFinishedSpy1(&database, &ChargingSessionsDatabase::databaseSessionUpdated);
    database.logEndSession(sessionId1, carId, carName, baseTime.addSecs(60));
    if (sessionFinishedSpy1.count() == 0) {
        sessionFinishedSpy1.wait();
    }
    QCOMPARE(sessionFinishedSpy1.count(), 1);

    // Session 2: baseTime+3600 .. baseTime+3660
    QSignalSpy sessionAddedSpy2(&database, &ChargingSessionsDatabase::databaseSessionAdded);
    database.logStartSession(evChargerId, evChargerName, serialNumber, carId, carName, baseTime.addSecs(3600));
    if (sessionAddedSpy2.count() == 0) {
        sessionAddedSpy2.wait();
    }
    QCOMPARE(sessionAddedSpy2.count(), 1);
    const int sessionId2 = sessionAddedSpy2.at(0).at(1).toInt();

    QSignalSpy sessionFinishedSpy2(&database, &ChargingSessionsDatabase::databaseSessionUpdated);
    database.logEndSession(sessionId2, carId, carName, baseTime.addSecs(3660));
    if (sessionFinishedSpy2.count() == 0) {
        sessionFinishedSpy2.wait();
    }
    QCOMPARE(sessionFinishedSpy2.count(), 1);

    // Fetch only session 2 by time range overlap
    FetchDataReply *fetchReply = database.fetchCarSessions(carId, baseTime.addSecs(3000), baseTime.addSecs(3700));
    QSignalSpy fetchReplySignalSpy(fetchReply, &FetchDataReply::finished);
    if (fetchReplySignalSpy.count() == 0) {
        fetchReplySignalSpy.wait();
    }
    QCOMPARE(fetchReplySignalSpy.count(), 1);
    QCOMPARE(fetchReply->error(), ChargingSessionsManager::ChargingSessionsErrorNoError);
    QCOMPARE(fetchReply->sessions().count(), 1);
    QCOMPARE(fetchReply->sessions().first().sessionId().toInt(), sessionId2);

    // Fetch all sessions overlapping from baseTime+30 onwards
    FetchDataReply *fetchReply2 = database.fetchCarSessions(carId, baseTime.addSecs(30), QDateTime());
    QSignalSpy fetchReply2SignalSpy(fetchReply2, &FetchDataReply::finished);
    if (fetchReply2SignalSpy.count() == 0) {
        fetchReply2SignalSpy.wait();
    }
    QCOMPARE(fetchReply2SignalSpy.count(), 1);
    QCOMPARE(fetchReply2->error(), ChargingSessionsManager::ChargingSessionsErrorNoError);
    QCOMPARE(fetchReply2->sessions().count(), 2);
}

QTEST_MAIN(ChargingSessionsDatabaseTests)
