// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-chargingsessions.
*
* nymea-energy-plugin-chargingsessions is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-chargingsessions is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-chargingsessions. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "chargingsessionsdatabase.h"
#include "chargingsessionssettings.h"
#include "chargingsessionsmanager.h"
#include "energymanagerdbusclient.h"
#include "processreply.h"

#include <QDir>
#include <QStandardPaths>
#include <QRegularExpression>
#include <QtConcurrent/QtConcurrent>


#include <loggingcategories.h>
NYMEA_LOGGING_CATEGORY(dcChargingSessions, "ChargingSessions")

ChargingSessionsManager::ChargingSessionsManager(EnergyManager *energyManager, ThingManager *thingManager, QObject *parent)
    : QObject{parent},
    m_energyManager{energyManager},
    m_thingManager{thingManager}
{
    qCDebug(dcChargingSessions()) << "Creating charging sessions manager";

    m_mailClient = new MailClient(this);

    m_database = new ChargingSessionsDatabase(ChargingSessionsSettings::databaseName(), this);
    connect(m_database, &ChargingSessionsDatabase::databaseSessionAdded, this, [this](const ThingId &evChargerId, const uint sessionId){

        ThingId carThingId = getAssociatedCarId(evChargerId);
        Thing *car = m_thingManager->findConfiguredThing(carThingId);
        Thing *evCharger = m_thingManager->findConfiguredThing(evChargerId);

        if (car) {
            qCDebug(dcChargingSessions()) << "Started charging session ID" << sessionId << "for" << evCharger ->name() << car->name();
        } else {
            qCWarning(dcChargingSessions()) << "Started charging session ID" << sessionId << "for" << evCharger->name() << "but there is no car associated yet.";
        }

        m_activeSessions[evCharger] = sessionId;
    });

    connect(m_database, &ChargingSessionsDatabase::databaseSessionUpdated, this, [this](int sessionId){

        Thing *evCharger = m_activeSessions.key(sessionId);
        if (!evCharger) {
            qCWarning(dcChargingSessions()) << "The charger for the updated session with ID" << sessionId << "could not be found any more. Ignoring event...";
            return;
        }

        ThingId carThingId = getAssociatedCarId(evCharger->id());
        Thing *car = m_thingManager->findConfiguredThing(carThingId);

        if (car) {
            qCDebug(dcChargingSessions()) << "Session with ID" << sessionId << "for" << evCharger->name() << car->name() << "updated successfully";
        } else {
            qCDebug(dcChargingSessions()) << "Session with ID" << sessionId << "for" << evCharger->name() << " updated successfully, but there is no car associated yet.";
        }
    });

    connect(m_database, &ChargingSessionsDatabase::databaseSessionFinished, this, [this](uint sessionId){
        if (!m_activeSessions.values().contains(sessionId))
            return;

        Thing *evCharger = m_activeSessions.key(sessionId);
        m_activeSessions.remove(evCharger);
        qCDebug(dcChargingSessions()) << "Session with ID" << sessionId << "for" << evCharger->name() << "cleaned up successfully";
    });

    ChargingSessionsSettings settings;
    qCDebug(dcChargingSessions()) << "Loading configuration from" << settings.fileName();
    m_configuration.setReporterName(settings.value("reporterName").toString());
    m_configuration.setReporterEmail(settings.value("reporterEmail").toString());
    m_configuration.setRecipientEmails(settings.value("recipientEmails").toStringList());

    // Initialize all ev charger things
    Things evChargers = m_thingManager->configuredThings().filterByInterface("evcharger");
    foreach (Thing *thing, evChargers) {
        onThingAdded(thing);
    }

    connect(m_thingManager, &ThingManager::thingAdded, this, &ChargingSessionsManager::onThingAdded);
    connect(m_thingManager, &ThingManager::thingRemoved, this, &ChargingSessionsManager::onThingRemoved);

    m_energyManagerClient = new EnergyManagerDbusClient(this);
    connect(m_energyManagerClient, &EnergyManagerDbusClient::chargingInfosUpdated, this, [](const QVariantList &chargingInfos){
        qCDebug(dcChargingSessions()) << "ChargingInfos:";
        foreach (const QVariant &ciVariant, chargingInfos) {
            qCDebug(dcChargingSessions()) << "-->" << ciVariant.toMap();
        }
    });

    connect(m_energyManagerClient, &EnergyManagerDbusClient::chargingInfoAdded, this, [](const QVariantMap &chargingInfo){
        qCDebug(dcChargingSessions()) << "ChargingInfo added:" << chargingInfo;
    });

    connect(m_energyManagerClient, &EnergyManagerDbusClient::chargingInfoChanged, this, [](const QVariantMap &chargingInfo){
        qCDebug(dcChargingSessions()) << "ChargingInfo changed:" << chargingInfo;
    });

    connect(m_energyManagerClient, &EnergyManagerDbusClient::chargingInfoRemoved, this, [](const QString &evChargerId){
        qCDebug(dcChargingSessions()) << "ChargingInfo removed:" << evChargerId;
    });

    connect(m_energyManagerClient, &EnergyManagerDbusClient::errorOccurred, this, [](const QString &errorMessage){
        qCWarning(dcChargingSessions()) << "Energy manager DBus client error:" << errorMessage;
    });

    qCDebug(dcChargingSessions()) << "ChargingInfos:" << m_energyManagerClient->chargingInfos();
    foreach (const QVariant &ciVariant, m_energyManagerClient->chargingInfos()) {
        qCDebug(dcChargingSessions()) << "-->" << ciVariant.toMap();
    }
}

ProcessReply *ChargingSessionsManager::sendReport(const QList<ThingId> carThingIds)
{
    if (carThingIds.isEmpty()) {
        qCDebug(dcChargingSessions()) << "Request to send report for all configured cars";
    } else {
        qCDebug(dcChargingSessions()) << "Request to send report for following car IDs" << carThingIds;
    }

    if (m_sendReportReply) {
        qCDebug(dcChargingSessions()) << "Requested to send a reply but there is already a reply pending. Re-using the same process.";
        return m_sendReportReply;
    }

    m_pendingFetchReplies.clear();
    m_pendingWriteJobs.clear();

    ProcessReply *processReply = new ProcessReply(this);
    processReply->m_startTimestamp = QDateTime::currentMSecsSinceEpoch();

    if (!m_configuration.isValid()) {
        qCWarning(dcChargingSessions()) << "Cannot send report. The configuration is incomplete" << m_configuration;
        processReply->finishReply(ChargingSessionsManager::ChargingSessionsErrorConfigurationIncomplete);
        return processReply;
    }

    // Generic car thing class
    Things configuredCars = m_thingManager->configuredThings().filterByInterface("electricvehicle");
    foreach (const ThingId &carId, carThingIds) {
        bool found = false;
        foreach (Thing *carThing, configuredCars) {
            if (carThing->id() == carId) {
                found = true;
                break;
            }
        }

        if (!found) {
            qCWarning(dcChargingSessions()) << "Cannot send report. There is no car configured with the given thing ID" << carId.toString();
            processReply->finishReply(ChargingSessionsManager::ChargingSessionsErrorUnknownCarThingId);
            return processReply;
        }
    }

    // Write report
    QDir storageDir(QStandardPaths::standardLocations(QStandardPaths::TempLocation).first() + QDir::separator() + "charging-sessions");
    if (storageDir.exists()) {
        if (!storageDir.removeRecursively()) {
            qCWarning(dcChargingSessions()) << "Unable to delete storage dir before creating reports in" << storageDir.absolutePath();
            processReply->finishReply(ChargingSessionsErrorInternalError);
            return processReply;
        } else {
            qCDebug(dcChargingSessions()) << "Cleaned up storage dir before creating new reports in" << storageDir.absolutePath();
        }
    }

    if (storageDir.mkpath(storageDir.path())) {
        qCDebug(dcChargingSessions()) << "Report storage dir created in" << storageDir.absolutePath();
    } else {
        qCWarning(dcChargingSessions()) << "Unable to create storage dir for saving reports in" << storageDir.absolutePath();
        processReply->finishReply(ChargingSessionsErrorInternalError);
        return processReply;
    }

    // Here we start the process queue
    m_sendReportReply = processReply;
    connect(processReply, &ProcessReply::destroyed, this, [this](){
        m_sendReportReply = nullptr;
    });

    // Get session data from db
    QString timeStamp = QDateTime::currentDateTime().toString("yyyyMMddhhmmss");

    if (carThingIds.isEmpty()) {

        FetchDataReply *reply = m_database->fetchCarSessions();
        connect(reply, &FetchDataReply::finished, this, [this, reply, storageDir, timeStamp](){

            QString reportFileName = storageDir.path() + QDir::separator() + QString("charging-sessions-report-%1.csv").arg(timeStamp);

            QFutureWatcher<bool> *watcher = writeCsvFile(reportFileName, reply->sessions());
            m_pendingWriteJobs.append(watcher);

            connect(watcher, &QFutureWatcher<bool>::finished, this, [this, watcher, reportFileName](){
                onWriteCsvFileFinished(reportFileName, watcher);
            });
        });
    } else {
        foreach (const ThingId &carId, carThingIds) {
            Thing *car = m_thingManager->findConfiguredThing(carId);
            qCDebug(dcChargingSessions()) << "Start generating report for" << car;
            FetchDataReply *reply = m_database->fetchCarSessions(carId);
            m_pendingFetchReplies.append(reply);
            connect(reply, &FetchDataReply::finished, this, [this, car, reply, storageDir, timeStamp](){

                m_pendingFetchReplies.removeAll(reply);

                QString reportFileName = storageDir.path() + QDir::separator() + QString("charging-sessions-report-%1-%2-%3.csv")
                                                                                     .arg(timeStamp)
                                                                                     .arg(car->name().replace(' ', '-'))
                                                                                     .arg(car->id().toString().remove('{').left(8));

                QFutureWatcher<bool> *watcher = writeCsvFile(reportFileName, reply->sessions());
                m_pendingWriteJobs.append(watcher);

                connect(watcher, &QFutureWatcher<bool>::finished, this, [this, watcher, reportFileName](){
                    onWriteCsvFileFinished(reportFileName, watcher);
                });
            });
        }
    }

    return processReply;
}

ChargingSessionsConfiguration ChargingSessionsManager::configuration() const
{
    return m_configuration;
}

ChargingSessionsManager::ChargingSessionsError ChargingSessionsManager::setConfiguration(const ChargingSessionsConfiguration &configuration)
{
    if (m_configuration == configuration)
        return ChargingSessionsManager::ChargingSessionsErrorNoError;

    static QRegularExpression regExp("\\b[A-Z0-9._%+-]+@[A-Z0-9.-]+\\.[A-Z]{2,62}\\b", QRegularExpression::CaseInsensitiveOption);

    if (!configuration.reporterEmail().isEmpty() && !regExp.match(configuration.reporterEmail()).hasMatch()) {
        qCWarning(dcChargingSessions()) << "The configuration contains an invalid reporter email address:" << configuration.reporterEmail();
        return ChargingSessionsManager::ChargingSessionsErrorInvalidEmail;
    }

    foreach (const QString &email, configuration.recipientEmails()) {
        if (!regExp.match(email).hasMatch()) {
            qCWarning(dcChargingSessions()) << "The configuration contains an invalid recipient email address:" << email;
            return ChargingSessionsManager::ChargingSessionsErrorInvalidEmail;
        }
    }

    qCDebug(dcChargingSessions()) << "Configuration changed:" << configuration;
    m_configuration = configuration;
    emit configurationChanged();

    ChargingSessionsSettings settings;
    qCDebug(dcChargingSessions()) << "Saving configuration to" << settings.fileName();
    settings.setValue("reporterName", configuration.reporterName());
    settings.setValue("reporterEmail", configuration.reporterEmail());
    settings.setValue("recipientEmails", configuration.recipientEmails());

    return ChargingSessionsManager::ChargingSessionsErrorNoError;
}

ChargingSessionsDatabase *ChargingSessionsManager::database() const
{
    return m_database;
}

void ChargingSessionsManager::onThingAdded(Thing *thing)
{
    if (verifyCharger(thing) && !m_evChargers.contains(thing)) {
        m_evChargers.append(thing);
        startMonitoringThingStates(thing);
    }
}

void ChargingSessionsManager::onThingRemoved(const ThingId &thingId)
{
    foreach (Thing *thing, m_evChargers) {
        if (thing->id() == thingId) {
            m_evChargers.removeAll(thing);
            stopMonitoringThingStates(thing);
        }
    }
}

void ChargingSessionsManager::onThingStateValueChanged(const StateTypeId &stateTypeId, const QVariant &value, const QVariant &minValue, const QVariant &maxValue, const QVariantList &possibleValues)
{
    Q_UNUSED(minValue)
    Q_UNUSED(maxValue)
    Q_UNUSED(possibleValues)

    Thing *thing = qobject_cast<Thing *>(sender());

    StateType stateType = thing->thingClass().stateTypes().findById(stateTypeId);
    if (stateType.name() == "pluggedIn") {
        bool pluggedIn = value.toBool();
        onEvChargerPluggedInChanged(thing, pluggedIn);
    } else if (stateType.name() == "sessionEnergy") {
        double sessionEnergy = value.toDouble();
        onEvChargerSessionEnergyChanged(thing, sessionEnergy);
    } else if (stateType.name() == "totalEnergyConsumed") {
        double totalEnergyConsumed = value.toDouble();
        onEvChargerTotalEnergyConsumedChanged(thing, totalEnergyConsumed);
    }

    // TODO: evaluate what to do on connected true/false

}

void ChargingSessionsManager::onEvChargerPluggedInChanged(Thing *evCharger, bool pluggedIn)
{
    ThingId carThingId = getAssociatedCarId(evCharger->id());
    Thing *car = m_thingManager->findConfiguredThing(carThingId);

    qCDebug(dcChargingSessions()) << "EV charger" << evCharger->name() << "session" << (pluggedIn ? "started" : "stopped") << QDateTime::currentDateTime().toString("dd.MM.yyyy hh:mm:ss");
    qCDebug(dcChargingSessions()) << "Associated Car:" << carThingId.toString();
    if (pluggedIn) {

        // Collect some data if possible...
        QString serialNumber;
        foreach (const Param &param, evCharger->params()) {
            if (evCharger->thingClass().paramTypes().findById(param.paramTypeId()).name().toLower() == "serialnumber") {
                serialNumber = param.value().toString();
            }
        }

        // TODO: if no serialnumber user the mac addresse

        double energyStart = 0;
        if (evCharger->hasState("totalEnergyConsumed"))
            energyStart = evCharger->stateValue("totalEnergyConsumed").toDouble(); // kWh

        ThingId carId;
        QString carName;
        if (car) {
            carId = car->id();
            carName = car->name();
        }

        m_database->logStartSession(evCharger->id(), evCharger->name(), serialNumber, carId, carName, QDateTime::currentDateTime(), energyStart);

    } else {

        // Plugged out
        if (!m_activeSessions.contains(evCharger)) {
            qCWarning(dcChargingSessions()) << "Could not finish session due to plugged out event because there is no active session for this charger.";
            return;
        }

        // Note: the active session will be removed when the session end has been logged successfully
        uint sessionId = m_activeSessions.value(evCharger);

        double energyEnd = 0;
        if (evCharger->hasState("totalEnergyConsumed"))
            energyEnd = evCharger->stateValue("totalEnergyConsumed").toDouble(); // kWh

        ThingId carId;
        QString carName;
        if (car) {
            carId = car->id();
            carName = car->name();
        }

        m_database->logEndSession(sessionId, carId, carName, QDateTime::currentDateTime(), energyEnd);
    }
}

void ChargingSessionsManager::onEvChargerSessionEnergyChanged(Thing *evCharger, double sessionEnergy)
{
    qCDebug(dcChargingSessions()) << "EV charger" << evCharger->name() << "session energy changed" << sessionEnergy;

    if (!m_activeSessions.contains(evCharger)) {
        // Fixme: evaluate if ending previouse session has to be closed or continue
        qCDebug(dcChargingSessions()) << "Received session energy but there is no active session for this charger.";
        return;
    }

    if (sessionEnergy <= 0) {
        qCDebug(dcChargingSessions()) << "Not writing the session energy" << sessionEnergy << "into the database because energy values <= 0 are not valid.";
        return;
    }

    m_database->updateSessionEnergy(m_activeSessions.value(evCharger), sessionEnergy, QDateTime::currentDateTime());
}

void ChargingSessionsManager::onEvChargerTotalEnergyConsumedChanged(Thing *evCharger, double totalEnergyConsumed)
{
    qCDebug(dcChargingSessions()) << "EV charger" << evCharger->name() << "total energy consumed energy changed" << totalEnergyConsumed;

    if (!m_activeSessions.contains(evCharger)) {
        // Fixme: evaluate if ending previouse session has to be closed or continue
        qCDebug(dcChargingSessions()) << "Received session energy but there is no active session for this charger.";
        return;
    }

    m_database->updateTotalEnergyConsumed(m_activeSessions.value(evCharger), totalEnergyConsumed, QDateTime::currentDateTime());
}

ThingId ChargingSessionsManager::getAssociatedCarId(const ThingId &evChargerId)
{
    foreach (const QVariant &chargingInfoVariant, m_energyManagerClient->chargingInfos()) {
        QVariantMap chargingInfo = chargingInfoVariant.toMap();
        if (chargingInfo.value("evChargerId").toUuid() == evChargerId) {
            if (!chargingInfo.value("assignedCarId").toString().isEmpty()) {
                return ThingId(chargingInfo.value("assignedCarId").toString());
            }
        }
    }

    return ThingId();
}

QFutureWatcher<bool> *ChargingSessionsManager::writeCsvFile(const QString &reportFileName, const QList<Session> &sessions)
{
    QFutureWatcher<bool> *watcher = new QFutureWatcher<bool>(this);
    QFuture<bool> future = QtConcurrent::run([reportFileName, sessions](){
        quint64 startTimestamp = QDateTime::currentMSecsSinceEpoch();
        if (QFileInfo::exists(reportFileName)) {
            qCWarning(dcChargingSessions()) << "Could not export data because the target csv file already exists" << reportFileName;
            return false;
        }

        QFile file(reportFileName);
        if (!file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate)) {
            qCWarning(dcChargingSessions()) << "Could not export data because the target file could not be opened:" << reportFileName << "error:" << file.error() << file.errorString();
            return false;
        }

        QTextStream textStream(&file);

        // Write data to the file
        QStringList headerColumns;
        headerColumns << QT_TR_NOOP("Session");
        headerColumns << QT_TR_NOOP("Session ID");
        headerColumns << QT_TR_NOOP("Charger name");
        headerColumns << QT_TR_NOOP("Charger serial number");
        headerColumns << QT_TR_NOOP("Car");
        headerColumns << QT_TR_NOOP("Start");
        headerColumns << QT_TR_NOOP("End");
        headerColumns << QT_TR_NOOP("Energy [kWh]");
        headerColumns << QT_TR_NOOP("Meter start [kWh]");
        headerColumns << QT_TR_NOOP("Meter end [kWh]");
        textStream << headerColumns.join(';') << '\n';

        uint rowCount = 1;

        foreach(const Session &session, sessions) {
            QStringList dataColums;
            dataColums << QString::number(rowCount);
            dataColums << session.sessionId();
            dataColums << session.chargerName().remove(';');
            dataColums << session.chargerSerialNumber().remove(';');
            dataColums << session.carName().remove(';');
            dataColums << session.startTimestamp().toString("yyyy.MM.dd hh:mm");
            dataColums << session.endTimestamp().toString("yyyy.MM.dd hh:mm");
            dataColums << QString::number(session.sessionEnergy()).replace(".",",");
            dataColums << QString::number(session.energyStart()).replace(".",",");
            dataColums << QString::number(session.energyEnd()).replace(".",",");

            qCDebug(dcChargingSessions()) << "Line" << dataColums;

            textStream << dataColums.join(';') << '\n';

            rowCount++;
        }

        file.close();
        qCDebug(dcChargingSessions()) << "CSV file written successfully"
                                      << QFileInfo(file.fileName()).size() << "[Bytes] in"
                                      << QDateTime::currentMSecsSinceEpoch() - startTimestamp << "ms";
        return true;
    });

    watcher->setFuture(future);
    return watcher;
}

void ChargingSessionsManager::onWriteCsvFileFinished(const QString &reportFileName, QFutureWatcher<bool> *watcher)
{
    // Clean up
    watcher->deleteLater();
    m_pendingWriteJobs.removeAll(watcher);

    // If there is no report reply any more, we are done
    if (!m_sendReportReply)
        return;

    if (!watcher->result()) {
        m_sendReportReply->finishReply(ChargingSessionsErrorInternalError);
        return;
    }

    // Add successfull written report file to list for email attachements
    m_reportFiles.append(reportFileName);

    // Check if this was the last job or if we have to wait
    if (m_pendingWriteJobs.isEmpty() && m_pendingFetchReplies.isEmpty()) {
        // Start sending email
        qCDebug(dcChargingSessions()).nospace() << "Successfully written " << reportFileName << ". This was the last write job. Continue with sending email ...";

        // Send email
        QString subject = QT_TR_NOOP("Charging sessions report");
        QString body = QT_TR_NOOP("New charging session report available!");
        m_networkReply = m_mailClient->sendEmail(m_configuration.reporterName(), m_configuration.reporterEmail(), m_configuration.recipientEmails(), subject, body, m_reportFiles);

        connect(m_networkReply, &QNetworkReply::finished, m_networkReply, &QNetworkReply::deleteLater);
        connect(m_networkReply, &QNetworkReply::finished, this, [this](){

            ChargingSessionsError error = ChargingSessionsErrorNoError;
            if (m_networkReply->error() != QNetworkReply::NoError) {
                error = ChargingSessionsErrorSendEmailFailed;
                qCWarning(dcChargingSessions()) << "Failed to send mail. Reply finished with error" << m_networkReply->error() << m_networkReply->errorString() << qUtf8Printable(m_networkReply->readAll());
            } else {
                qCDebug(dcChargingSessions()) << "Sent email successfully" << m_networkReply->error() << m_networkReply->errorString() << qUtf8Printable(m_networkReply->readAll());
            }

            qCDebug(dcChargingSessions()) << "Send report process finished after" << (QDateTime::currentMSecsSinceEpoch() - m_sendReportReply->m_startTimestamp) << "ms.";
            m_sendReportReply->finishReply(error);
        });

    } else {
        qCDebug(dcChargingSessions()).nospace() << "Successfully written " << reportFileName << ". There are jobs to do. Fetch data jobs:"
                                                << m_pendingFetchReplies.count() << " Write data jobs: " << m_pendingWriteJobs.count();
    }
}

bool ChargingSessionsManager::verifyCharger(Thing *thing)
{
    // Make sure this is an evcharger
    if (!thing->thingClass().interfaces().contains("evcharger"))
        return false;

    // Make sure we have some energy information
    if (!thing->hasState("sessionEnergy") && !thing->hasState("totalEnergyConsumed"))
        return false;

    // Make sure we have everything in order to recognize the session start/stop
    if (!thing->hasState("pluggedIn"))
        return false;

    return true;
}

void ChargingSessionsManager::startMonitoringThingStates(Thing *thing)
{
    qCDebug(dcChargingSessions()) << "Start monitoring charging sessions for" << thing;
    connect(thing, &Thing::stateValueChanged, this, &ChargingSessionsManager::onThingStateValueChanged);
}

void ChargingSessionsManager::stopMonitoringThingStates(Thing *thing)
{
    qCDebug(dcChargingSessions()) << "Stop monitoring charging sessions for" << thing;
    disconnect(thing, &Thing::stateValueChanged, this, &ChargingSessionsManager::onThingStateValueChanged);
}
