// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-chargingsessions.
*
* nymea-energy-plugin-chargingsessions is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-chargingsessions is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-chargingsessions. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "chargingsessionsdatabase.h"

#include <QFile>
#include <QSqlError>
#include <QSqlQuery>
#include <QFileInfo>
#include <QDateTime>
#include <QRegularExpression>
#include <QFuture>
#include <QtConcurrent/QtConcurrent>

#include <QLoggingCategory>
Q_DECLARE_LOGGING_CATEGORY(dcChargingSessions)

ChargingSessionsDatabase::ChargingSessionsDatabase(const QString &databaseName, QObject *parent)
    : QObject{parent},
      m_databaseName{databaseName}
{
    QFileInfo databaseFileInfo(m_databaseName);
    QDir storageDir = QDir(databaseFileInfo.absolutePath());

    if (!storageDir.exists()) {
        if (!storageDir.mkpath(storageDir.absolutePath())) {
            qCWarning(dcChargingSessions()) << "Unable create storage dir" << storageDir.absolutePath();
            return;
        }
    }

    m_connectionName = databaseFileInfo.baseName();
    m_db = QSqlDatabase::addDatabase(QStringLiteral("QSQLITE"), m_connectionName);
    m_db.setDatabaseName(m_databaseName);

    if (!m_db.isValid()) {
        qCWarning(dcChargingSessions()) << "The database is not valid" << m_db.databaseName();
        // FIXME: rotate database
        return;
    }

    qCDebug(dcChargingSessions()) << "Opening database" << m_db.databaseName();
    if (!initDatabase()) {
        qCWarning(dcChargingSessions()) << "Failed to initialize the database" << m_db.databaseName();
        // FIXME: rotate database
        return;
    }

    qCDebug(dcChargingSessions()) << "Database initialized successfully.";
    m_initialized = true;

    connect(&m_jobWatcher, &QFutureWatcher<DatabaseJob*>::finished, this, &ChargingSessionsDatabase::handleJobFinished);
}

ChargingSessionsDatabase::~ChargingSessionsDatabase()
{
    // Process the job queue before allowing to shut down
    while (m_currentJob) {
        qCDebug(dcChargingSessions()) << "Waiting for job to finish... (" << m_jobQueue.count() << "jobs left in queue)";
        m_jobWatcher.waitForFinished();
        // Make sure that the job queue is processes
        // We can't call processQueue ourselves because thread synchronisation is done via queued connections
        qApp->processEvents();
    }
    qCDebug(dcChargingSessions()) << "Closing Database";

    if (m_db.isOpen()) {
        m_db.close();
    }
    m_db = QSqlDatabase();
    QSqlDatabase::removeDatabase(m_connectionName);
}

QString ChargingSessionsDatabase::databaseName() const
{
    return m_databaseName;
}

void ChargingSessionsDatabase::enqueJob(DatabaseJob *job)
{
    m_jobQueue.append(job);
    qCDebug(dcChargingSessions()).nospace() << "Scheduled job at position " << (m_jobQueue.count() - 1) << " (" << m_jobQueue.count() << " jobs in the queue)";
    processQueue();
}

void ChargingSessionsDatabase::processQueue()
{
    if (!m_initialized) {
        return;
    }

    if (m_jobQueue.isEmpty()) {
        return;
    }

    if (m_currentJob) {
        return;
    }

    DatabaseJob *job = nullptr;
    job = m_jobQueue.takeFirst();
    qCDebug(dcChargingSessions()).nospace() << "Processing DB queue. (" << m_jobQueue.count() << " jobs left in queue)";
    job->m_startTimestamp = QDateTime::currentMSecsSinceEpoch();

    m_currentJob = job;

    QFuture<DatabaseJob*> future = QtConcurrent::run([job, this](){
        QSqlQuery query(job->m_db);
        query.prepare(job->m_queryString);

        query.exec();

        job->m_error = query.lastError();
        job->m_executedQuery = query.executedQuery();

        if (!query.lastError().isValid()) {
            while (query.next()) {
                job->m_results.append(query.record());
            }
        }
        m_lastSessionId = query.lastInsertId().toUInt();
        return job;
    });

    m_jobWatcher.setFuture(future);
}

void ChargingSessionsDatabase::handleJobFinished()
{
    DatabaseJob *job = m_jobWatcher.result();
    emit job->finished();
    quint64 duration = QDateTime::currentMSecsSinceEpoch() - job->m_startTimestamp;
    job->deleteLater();
    m_currentJob = nullptr;

    qCDebug(dcChargingSessions()) << "DB job finished (Duration" << duration << "ms)";
    processQueue();
}

Session ChargingSessionsDatabase::parseSession(const QSqlRecord &result)
{
    Session session;
    session.m_sessionId = QString::number(result.value("id").toInt());
    session.m_chargerName = result.value("chargerName").toString().remove(';');
    session.m_chargerSerialNumber = result.value("chargerSerialNumber").toString().remove(';');
    session.m_carName = result.value("carName").toString().remove(';');
    session.m_startTimestamp = QDateTime::fromSecsSinceEpoch(result.value("startTimestamp").toLongLong());
    session.m_endTimestamp = QDateTime::fromSecsSinceEpoch(result.value("endTimestamp").toLongLong());
    session.m_sessionEnergy = result.value("sessionEnergy").toDouble();
    session.m_energyStart = result.value("energyStart").toDouble();
    session.m_energyEnd = result.value("energyEnd").toDouble();
    return session;
}

void ChargingSessionsDatabase::logStartSession(const ThingId &evChargerId, const QString &evChargerName, const QString &serialNumber, const ThingId &carId, const QString &carName, const QDateTime &startDateTime, double energyStart)
{
    qCDebug(dcChargingSessions()) << "--> Logging the start of the session in the database";
    QString queryString = QString("INSERT OR REPLACE INTO chargingSessions (chargerUuid, chargerName, chargerSerialNumber, carUuid, carName, startTimestamp, lastUpdate, energyStart, sessionEnergy) "
                                  "VALUES (\"%1\", \"%2\", \"%3\", \"%4\", \"%5\", \"%6\", \"%7\", \"%8\", \"%9\");")
            .arg(evChargerId.toString().remove('{').remove('}'))
            .arg(evChargerName)
            .arg(serialNumber)
            .arg(!carId.isNull() ? carId.toString().remove('{').remove('}') : QString()) // Note: car might not be associated yet or unknwon
            .arg(carName)
            .arg(startDateTime.toSecsSinceEpoch())
            .arg(startDateTime.toSecsSinceEpoch())
            .arg(energyStart)
            .arg(0);

    qCDebug(dcChargingSessions()) << qUtf8Printable(queryString);

    DatabaseJob *job = new DatabaseJob(m_db, queryString);
    connect(job, &DatabaseJob::finished, this, [this, job, evChargerId](){

        if (job->error().type() != QSqlError::NoError) {
            qCWarning(dcChargingSessions) << "Error log session start. Driver error:" << job->error().driverText() << "Database error:" << job->error().databaseText();
            return;
        }

        qCDebug(dcChargingSessions()) << "Logged successfully new charging session start: session ID" << m_lastSessionId;
        emit databaseSessionAdded(evChargerId, m_lastSessionId);
    });
    enqueJob(job);
}

void ChargingSessionsDatabase::logEndSession(int sessionId, const ThingId &carId, const QString &carName, const QDateTime &endDateTime, double energyEnd)
{
    qCDebug(dcChargingSessions()) << "--> Logging the end of session" << sessionId << "in the database";

    FetchDataReply *fetchReply = fetchRow(sessionId);
    connect(fetchReply, &FetchDataReply::finished, this, [fetchReply, sessionId, carId, carName, endDateTime, energyEnd, this](){
        if (fetchReply->error() != ChargingSessionsManager::ChargingSessionsErrorNoError) {
            qCWarning(dcChargingSessions()) << "Unable to fetch row for session ID" << sessionId;
            return;
        }

        if (fetchReply->sessions().isEmpty()) {
            qCWarning(dcChargingSessions()) << "Found no matching session with session ID" << sessionId;
            return;
        }

        Session session = fetchReply->sessions().first();
        double sessionEnergy = session.sessionEnergy();
        if (energyEnd != 0 && session.energyStart() != 0) {
            sessionEnergy = qRound((energyEnd - session.energyStart()) * 10000) / 10000.0;
            qCDebug(dcChargingSessions()) << "Calculated session energy from start:" << session.energyStart() << "end:" << energyEnd << "=" << sessionEnergy << "kWh";
        }
        QString queryString = QString("UPDATE chargingSessions SET lastUpdate = \"%1\", carUuid = \"%2\", carName = \"%3\", endTimestamp = \"%4\", energyEnd = \"%5\", sessionEnergy = \"%6\" WHERE id = \"%7\";")
                                  .arg(endDateTime.toSecsSinceEpoch())
                                  .arg(carId.toString().remove('{').remove('}'))
                                  .arg(carName)
                                  .arg(endDateTime.toSecsSinceEpoch())
                                  .arg(energyEnd)
                                  .arg(sessionEnergy)
                                  .arg(sessionId);

        qCDebug(dcChargingSessions()) << qUtf8Printable(queryString);

        DatabaseJob *updateJob = new DatabaseJob(m_db, queryString);
        connect(updateJob, &DatabaseJob::finished, this, [sessionId, updateJob, this](){

            if (updateJob->error().type() != QSqlError::NoError) {
                qCWarning(dcChargingSessions) << "Error log session end. Driver error:" << updateJob->error().driverText() << "Database error:" << updateJob->error().databaseText();
                return;
            }

            qCDebug(dcChargingSessions()) << "Logged successfully the end of the charging session with ID" << sessionId;
            emit databaseSessionUpdated(sessionId);
            emit databaseSessionFinished(sessionId);
        });
        enqueJob(updateJob);
    });
}

FetchDataReply *ChargingSessionsDatabase::fetchCarSessions(const ThingId &carThingId, const QDateTime &startDateTime, const QDateTime &endDateTime)
{
    FetchDataReply *reply = new FetchDataReply(this);

    QStringList conditions;
    conditions << QStringLiteral("endTimestamp IS NOT NULL");

    if (!carThingId.isNull()) {
        conditions << QString("carUuid = \"%1\"").arg(carThingId.toString().remove('{').remove('}'));
    }

    if (startDateTime.isValid() && endDateTime.isValid()) {
        const qint64 startTimestamp = startDateTime.toSecsSinceEpoch();
        const qint64 endTimestamp = endDateTime.toSecsSinceEpoch();
        conditions << QString("startTimestamp <= \"%1\"").arg(endTimestamp);
        conditions << QString("endTimestamp >= \"%1\"").arg(startTimestamp);
    } else if (startDateTime.isValid()) {
        const qint64 startTimestamp = startDateTime.toSecsSinceEpoch();
        conditions << QString("endTimestamp >= \"%1\"").arg(startTimestamp);
    } else if (endDateTime.isValid()) {
        const qint64 endTimestamp = endDateTime.toSecsSinceEpoch();
        conditions << QString("startTimestamp <= \"%1\"").arg(endTimestamp);
    }

    const QString queryString = QString("SELECT * FROM chargingSessions WHERE %1 ORDER BY startTimestamp DESC;").arg(conditions.join(QStringLiteral(" AND ")));

    qCDebug(dcChargingSessions()) << qUtf8Printable(queryString);

    DatabaseJob *fetchJob = new DatabaseJob(m_db, queryString);
    connect(fetchJob, &DatabaseJob::finished, this, [queryString, fetchJob, reply, this](){

        if (fetchJob->error().type() != QSqlError::NoError) {
            qCWarning(dcChargingSessions()) << "Could read report from database." << queryString << m_db.lastError().databaseText() << m_db.lastError().driverText();
            reply->finishReply(ChargingSessionsManager::ChargingSessionsErrorInternalError);
            return;
        }

        foreach(const QSqlRecord &result, fetchJob->results()) {
            reply->m_sessions.append(parseSession(result));
        }

        reply->finishReply();
    });
    enqueJob(fetchJob);

    return reply;
}


FetchDataReply *ChargingSessionsDatabase::fetchRow(uint sessionId)
{
    FetchDataReply *reply = new FetchDataReply(this);

    // Fetch data from db
    QString queryString = QString("SELECT * FROM chargingSessions WHERE id == \"%1\";").arg(sessionId);

    qCDebug(dcChargingSessions()) << qUtf8Printable(queryString);

    DatabaseJob *fetchJob = new DatabaseJob(m_db, queryString);
    connect(fetchJob, &DatabaseJob::finished, this, [queryString, fetchJob, reply, this](){

        if (fetchJob->error().type() != QSqlError::NoError) {
            qCWarning(dcChargingSessions()) << "Could read report from database." << queryString << m_db.lastError().databaseText() << m_db.lastError().driverText();
            reply->finishReply(ChargingSessionsManager::ChargingSessionsErrorInternalError);
            return;
        }

        foreach(const QSqlRecord &result, fetchJob->results()) {
            reply->m_sessions.append(parseSession(result));
        }

        reply->finishReply();
    });
    enqueJob(fetchJob);

    return reply;
}

void ChargingSessionsDatabase::updateSessionEnergy(uint sessionId, double sessionEnergy, const QDateTime &lastUpdate)
{
    QString queryString = QString("UPDATE chargingSessions SET lastUpdate = \"%1\", sessionEnergy = \"%2\" WHERE id = \"%3\";")
            .arg(lastUpdate.toSecsSinceEpoch())
            .arg(sessionEnergy)
            .arg(sessionId);

    qCDebug(dcChargingSessions()) << qUtf8Printable(queryString);

    DatabaseJob *updateJob = new DatabaseJob(m_db, queryString);
    connect(updateJob, &DatabaseJob::finished, this, [sessionId, updateJob, this](){

        if (updateJob->error().type() != QSqlError::NoError) {
            qCWarning(dcChargingSessions) << "Error update session energy. Driver error:" << updateJob->error().driverText() << "Database error:" << updateJob->error().databaseText();
            return;
        }

        qCDebug(dcChargingSessions()) <<  "Updated the session energy successfully in the database with ID" << sessionId;
        emit databaseSessionUpdated(sessionId);
    });
    enqueJob(updateJob);
}

void ChargingSessionsDatabase::updateTotalEnergyConsumed(uint sessionId, double totalEnergyConsumed, const QDateTime &lastUpdate)
{
    QString queryString = QString("UPDATE chargingSessions SET lastUpdate = \"%1\", energyEnd = \"%2\" WHERE id = \"%3\";")
            .arg(lastUpdate.toSecsSinceEpoch())
            .arg(totalEnergyConsumed)
            .arg(sessionId);

    qCDebug(dcChargingSessions()) << qUtf8Printable(queryString);

    DatabaseJob *updateJob = new DatabaseJob(m_db, queryString);
    connect(updateJob, &DatabaseJob::finished, this, [this, sessionId, updateJob](){

        if (updateJob->error().type() != QSqlError::NoError) {
            qCWarning(dcChargingSessions) << "Error update session energy. Driver error:" << updateJob->error().driverText() << "Database error:" << updateJob->error().databaseText();
            return;
        }

        qCDebug(dcChargingSessions()) << "Updated the totlal energy consumed successfully in the database";
        emit databaseSessionUpdated(sessionId);
    });
    enqueJob(updateJob);
}

bool ChargingSessionsDatabase::wipeDatabase()
{
    qCDebug(dcChargingSessions()) << "Wipe all database entries from" << m_db.databaseName();

    QSqlQuery deleteQuery(m_db);
    if (!deleteQuery.exec("DELETE FROM chargingSessions;")) {
        qCWarning(dcChargingSessions()) << "Unable to execute SQL query" << deleteQuery.lastQuery() << m_db.lastError().databaseText() << m_db.lastError().driverText();
        return false;
    }

    if (m_db.lastError().type() != QSqlError::NoError) {
        qCWarning(dcChargingSessions()) << "Could not delete all charging session." << m_db.lastError().databaseText() << m_db.lastError().driverText();
        return false;
    }

    m_db.close();
    m_db = QSqlDatabase();
    QSqlDatabase::removeDatabase(m_connectionName);

    // Delete database file
    QFile databaseFile(m_databaseName);
    if (databaseFile.exists()) {
        if (!databaseFile.remove()) {
            qCWarning(dcChargingSessions()) << "Could not delete database file" << m_databaseName;
            return false;
        }
    }

    return true;
}

bool ChargingSessionsDatabase::initDatabase()
{
    // Reopen the db to make sure it can be written
    m_db.close();
    if (!m_db.open()) {
        qCWarning(dcChargingSessions()) << "Could not open the charging sessions database" << m_db.databaseName() << m_db.lastError();
        return false;
    }

    qCDebug(dcChargingSessions()) << "Tables" << m_db.tables();
    if (m_db.tables().isEmpty()) {
        // Write pragmas
        QSqlQuery enableForeigenKeysQuery("PRAGMA foreign_keys = ON;", m_db);
        if (!enableForeigenKeysQuery.exec()) {
            qCWarning(dcChargingSessions()) << "Unable to execute SQL query" << enableForeigenKeysQuery.lastQuery() << m_db.lastError().databaseText() << m_db.lastError().driverText();
            return false;
        }

        QSqlQuery setUserVersionQuery(QString("PRAGMA user_version = %1;").arg(DB_VERSION), m_db);
        if (!setUserVersionQuery.exec()) {
            qCWarning(dcChargingSessions()) << "Unable to execute SQL query" << setUserVersionQuery.lastQuery() << m_db.lastError().databaseText() << m_db.lastError().driverText();
            return false;
        }
    } else {
        QSqlQuery getUserVersionQuery(QString("PRAGMA user_version;"), m_db);
        if (!getUserVersionQuery.exec()) {
            qCWarning(dcChargingSessions()) << "Unable to execute SQL query" << getUserVersionQuery.lastQuery() << m_db.lastError().databaseText() << m_db.lastError().driverText();
        } else {
            while (getUserVersionQuery.next()) {
                m_currentDbVersion = getUserVersionQuery.value("user_version").toLongLong();
                qCDebug(dcChargingSessions()) << "The current database version is" << m_currentDbVersion;
            }
        }
    }

    // Note: once we need migration, check schema version (PRAGMA user_version) for compatibility here and perform migrations if required

    // Create nodes table
    if (!m_db.tables().contains("chargingSessions")) {
        bool tableCreated = createTable("chargingSessions",
                                        "(id INTEGER PRIMARY KEY AUTOINCREMENT, " // unique session ID
                                        "chargerUuid TEXT NOT NULL, " // uuid from the EV charger
                                        "chargerName TEXT, " // User given name of the EV charger
                                        "chargerSerialNumber TEXT, " // Serial number of the EV charger if known
                                        "carUuid TEXT NOT NULL, " // uuid from the associated car
                                        "carName TEXT, " // User given name of the car
                                        "startTimestamp INTEGER NOT NULL, " // unix timestamp of the session start
                                        "endTimestamp INTEGER, " // unix timestamp of the session end
                                        "lastUpdate INTEGER NOT NULL, " // unix timestamp when this row has been updated the last time
                                        "sessionEnergy REAL, " // total energy loaded in this session kWh
                                        "energyStart REAL, " // meter energy on the start of the session kWh
                                        "energyEnd REAL" // meter energy on the end of the session kWh
                                        ")");

        if (!tableCreated) {
            return false;
        }
    }

    return true;
}

bool ChargingSessionsDatabase::createTable(const QString &tableName, const QString &schema)
{
    qCDebug(dcChargingSessions()) << "Creating table" << tableName << schema;
    QString queryString = QString("CREATE TABLE IF NOT EXISTS %1 %2;").arg(tableName).arg(schema);
    QSqlQuery query(m_db);
    if (!query.exec(queryString)) {
        qCWarning(dcChargingSessions()) << "Failed to execute query" << queryString << query.lastError();
        return false;
    }

    if (m_db.lastError().type() != QSqlError::NoError) {
        qCWarning(dcChargingSessions()) << "Could not create table in database." << queryString << m_db.lastError().databaseText() << m_db.lastError().driverText();
        return false;
    }

    return true;
}
