// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-chargingsessions.
*
* nymea-energy-plugin-chargingsessions is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-chargingsessions is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-chargingsessions. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "mailclient.h"

#include <QHttpMultiPart>
#include <QFile>
#include <QFileInfo>
#include <QNetworkReply>
#include <QUrlQuery>

#include <QLoggingCategory>
Q_DECLARE_LOGGING_CATEGORY(dcChargingSessions)

MailClient::MailClient(QObject *parent)
    : QObject{parent},
    m_networkAccessManager{new QNetworkAccessManager(this)}
{

}

QNetworkReply *MailClient::sendEmail(const QString &senderName, const QString &senderEmail, const QStringList recipientEmails, const QString &subject, const QString &body, const QStringList attachments)
{
    qCDebug(dcChargingSessions()) << "Sending reports as" << senderName << QString("<%1>").arg(senderEmail) << "to" << recipientEmails << "subject:" << subject;;

    QHttpMultiPart *multiPart = new QHttpMultiPart(QHttpMultiPart::FormDataType);

    QHttpPart fromPart;
    fromPart.setHeader(QNetworkRequest::ContentDispositionHeader, "form-data; name=\"from\"");
    fromPart.setBody(senderName.toUtf8() + "<" + senderEmail.toUtf8() + ">");
    multiPart->append(fromPart);

    foreach (const QString &recipient, recipientEmails) {
        QHttpPart toPart;
        toPart.setHeader(QNetworkRequest::ContentDispositionHeader, "form-data; name=\"to\"");
        toPart.setBody(recipient.toUtf8());
        multiPart->append(toPart);
    }

    QHttpPart subjectPart;
    subjectPart.setHeader(QNetworkRequest::ContentDispositionHeader, "form-data; name=\"subject\"");
    subjectPart.setBody(subject.toUtf8());
    multiPart->append(subjectPart);

    QHttpPart textPart;
    textPart.setHeader(QNetworkRequest::ContentDispositionHeader, "form-data; name=\"text\"");
    textPart.setBody(body.toUtf8());
    multiPart->append(textPart);

    QHttpPart replyPart;
    replyPart.setHeader(QNetworkRequest::ContentDispositionHeader, "form-data; name=\"h:reply-To\"");
    replyPart.setBody(senderEmail.toUtf8());
    multiPart->append(replyPart);

    foreach (const QString &attachment, attachments) {
        QFile *file = new QFile(QString(attachment).remove("file://"));
        file->setParent(multiPart);
        if (file->open(QIODevice::ReadOnly)) {
            QFileInfo attachmentFileInfo(attachment);
            QString headerLine = QString("form-data; name=\"attachment\"; filename=\"%1\"; contentType=\"text/csv\"").arg(attachmentFileInfo.fileName());
            qCDebug(dcChargingSessions()) << "Attache file" << attachment << headerLine;

            QHttpPart imagePart;
            imagePart.setHeader(QNetworkRequest::ContentDispositionHeader, QVariant(headerLine));
            imagePart.setBodyDevice(file);
            multiPart->append(imagePart);
        }
    }

    QUrl url("https://api.eu.mailgun.net/v3/sendmail.nymea.io/messages");

    QNetworkRequest request(url);
    QByteArray credentials("api:key-6da6a80f4d965d0c168114d1d665a875");
    request.setRawHeader("Authorization", "Basic " + credentials.toBase64());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "multipart/form-data; boundary=" + multiPart->boundary());

    QNetworkReply *reply = m_networkAccessManager->post(request, multiPart);
    multiPart->setParent(reply);

    return reply;
}
