// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-chargingsessions.
*
* nymea-energy-plugin-chargingsessions is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-chargingsessions is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-chargingsessions. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "chargingsessionstestbase.h"

ChargingSessionsTestBase::ChargingSessionsTestBase(QObject *parent) :
    NymeaTestBase(parent)
{
    qCDebug(dcTests()) << "Running in:" << QDir::currentPath();

    QString energyPluginEnvPath = QDir(QDir::currentPath() + "/../../../energyplugin/").absolutePath();
    qCDebug(dcTests()) << "Energy plugin path:" << energyPluginEnvPath;

    // Add this enery plugin to the paths
    qputenv("NYMEA_ENERGY_PLUGINS_EXTRA_PATH", energyPluginEnvPath.toUtf8());

    qCDebug(dcTests()).nospace() << "NYMEA_ENERGY_PLUGINS_EXTRA_PATH=" << qgetenv("NYMEA_ENERGY_PLUGINS_EXTRA_PATH");
    qCDebug(dcTests()).nospace() << "NYMEA_EXPERIENCE_PLUGINS_PATH=" << qgetenv("NYMEA_EXPERIENCE_PLUGINS_PATH");
}

void ChargingSessionsTestBase::initTestCase(const QString &loggingRules, bool checkExperienceLoaded, bool disableLogEngine)
{
    QDir dir(NymeaSettings::translationsPath());
    dir.mkpath(NymeaSettings::translationsPath());
    QStringList languages = {"de", "en_US"};
    foreach (const QString &language, languages) {
        QFile f(NymeaSettings::translationsPath().append("/nymead-" + language + ".qm"));
        QVERIFY2(f.open(QFile::WriteOnly), "Could not create translation file.");
        f.write(" ");
        f.close();
    }

    // If testcase asserts cleanup won't do. Lets clear any previous test run settings leftovers
    QSettings energySettings(NymeaSettings::settingsPath() +  "/chargingsessions.conf", QSettings::IniFormat);
    energySettings.clear();

    if (loggingRules.isEmpty()) {
        NymeaTestBase::initTestCase("*.info=true\n*.debug=false\nApplication.debug=true\nTests.debug=true\nLogEngine.info=false\nExperiences.debug=true\nChargingSessions.debug=true\nEnergyMocks.debug=true\nDBus.warning=false", disableLogEngine);
    } else {
        NymeaTestBase::initTestCase(loggingRules, disableLogEngine);
    }

    QVariant reply = injectAndWait("JSONRPC.Hello");
    qCDebug(dcTests()) << qUtf8Printable(QJsonDocument::fromVariant(reply).toJson());

    if (checkExperienceLoaded) {
        QString experienceVersion = "0.1"; // TODO: Perhaps this should be defined in the project file somewhere
        QVERIFY2(reply.toMap().value("params").toMap().contains("experiences"),
                 QString("No experience plugins loaded!\n%1")
                 .arg(QString(QJsonDocument::fromVariant(reply).toJson())).toUtf8());
        QVariantMap experienceDefinition;
        experienceDefinition.insert("name", "ChargingSessions");
        experienceDefinition.insert("version", experienceVersion);

        QVERIFY2(reply.toMap().value("params").toMap().value("experiences").toList().contains(experienceDefinition),
                 QString("NymeaEnergy %1 experience plugins not loaded!\n%2")
                 .arg(experienceVersion)
                 .arg(QString(QJsonDocument::fromVariant(reply).toJson())).toUtf8());

        enableNotifications({"Integrations", "Energy", "ChargingSessions"});
    }

    removeDevices();
}

void ChargingSessionsTestBase::cleanupTestCase()
{
    removeDevices();
    NymeaTestBase::cleanupTestCase();
}

void ChargingSessionsTestBase::init()
{
    removeDevices();
}

void ChargingSessionsTestBase::cleanup()
{
    removeDevices();
}

void ChargingSessionsTestBase::removeDevices()
{
    QVariant configuredDevices = injectAndWait("Integrations.GetThings");
    foreach (const QVariant &dev, configuredDevices.toMap().value("params").toMap().value("things").toList()) {
        qCDebug(dcTests()) << "Removing thing" << dev.toMap().value("name").toString() << dev.toMap().value("id").toUuid();
        QVariant response = removeDevice(dev.toMap().value("id").toUuid());
        verifyThingError(response);
    }
}

QVariant ChargingSessionsTestBase::removeDevice(const QUuid &thingId)
{
    qCDebug(dcTests()) << "Remove device" << thingId.toString();
    QVariantMap params;
    params.insert("thingId", thingId.toString());
    return injectAndWait("Integrations.RemoveThing", params);
}
