/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright 2013 - 2024, nymea GmbH
* Contact: contact@nymea.io
*
* This file is part of nymea.
* This project including source code and documentation is protected by
* copyright law, and remains the property of nymea GmbH. All rights, including
* reproduction, publication, editing and translation, are reserved. The use of
* this project is subject to the terms of a license agreement to be concluded
* with nymea GmbH in accordance with the terms of use of nymea GmbH, available
* under https://nymea.io/license
*
* GNU Lesser General Public License Usage
* Alternatively, this project may be redistributed and/or modified under the
* terms of the GNU Lesser General Public License as published by the Free
* Software Foundation; version 3. This project is distributed in the hope that
* it will be useful, but WITHOUT ANY WARRANTY; without even the implied
* warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with this project. If not, see <https://www.gnu.org/licenses/>.
*
* For any further details and any questions please contact us under
* contact@nymea.io or see our FAQ/Licensing Information on
* https://nymea.io/license/faq
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef CHARGINGSESSIONSMANAGER_H
#define CHARGINGSESSIONSMANAGER_H

#include <QObject>

// from libnymea
#include <integrations/thingmanager.h>

// from libnymea-energy
#include <energymanager.h>

#include "mailclient.h"
#include "chargingsessionsconfiguration.h"

#include <QHash>
#include <QFuture>

class Session;
class ProcessReply;
class FetchDataReply;
class ChargingSessionsDatabase;

class ChargingSessionsManager : public QObject
{
    Q_OBJECT
public:
    enum ChargingSessionsError {
        ChargingSessionsErrorNoError,
        ChargingSessionsErrorInvalidEmail,
        ChargingSessionsErrorUnknownCarThingId,
        ChargingSessionsErrorConfigurationIncomplete,
        ChargingSessionsErrorSendEmailFailed,
        ChargingSessionsErrorInternalError
    };
    Q_ENUM(ChargingSessionsError)

    explicit ChargingSessionsManager(EnergyManager *energyManager, ThingManager *thingManager, QObject *parent = nullptr);

    ProcessReply *sendReport(const QList<ThingId> carThingIds);

    ChargingSessionsConfiguration configuration() const;
    ChargingSessionsManager::ChargingSessionsError setConfiguration(const ChargingSessionsConfiguration &configuration);

signals:
    void configurationChanged();

private slots:
    void onThingAdded(Thing *thing);
    void onThingRemoved(const ThingId &thingId);
    void onThingStateValueChanged(const StateTypeId &stateTypeId, const QVariant &value, const QVariant &minValue, const QVariant &maxValue, const QVariantList &possibleValues);

    void onEvChargerPluggedInChanged(Thing *evCharger, bool pluggedIn);
    void onEvChargerSessionEnergyChanged(Thing *evCharger, double sessionEnergy);
    void onEvChargerTotalEnergyConsumedChanged(Thing *evCharger, double totalEnergyConsumed);

private:
    EnergyManager *m_energyManager = nullptr;
    ThingManager *m_thingManager = nullptr;

    MailClient *m_mailClient = nullptr;
    ChargingSessionsDatabase *m_database = nullptr;
    ChargingSessionsConfiguration m_configuration;

    Things m_evChargers;

    QHash<uint, double> m_sessionStartEnergy;
    QHash<Thing *, uint> m_activeSessions;

    QStringList m_reportFiles;
    ProcessReply *m_sendReportReply = nullptr;
    QNetworkReply *m_networkReply = nullptr;;

    QList<FetchDataReply *> m_pendingFetchReplies;
    QList<QFutureWatcher<bool> *> m_pendingWriteJobs;

    QFutureWatcher<bool> *writeCsvFile(const QString &reportFileName, const QList<Session> &sessions);
    void onWriteCsvFileFinished(const QString &reportFileName, QFutureWatcher<bool> *watcher);

    bool verifyCharger(Thing *thing);

    void startMonitoringThingStates(Thing *thing);
    void stopMonitoringThingStates(Thing *thing);
};


#endif // CHARGINGSESSIONSMANAGER_H
