/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright 2013 - 2022, nymea GmbH
* Contact: contact@nymea.io
*
* This file is part of nymea.
* This project including source code and documentation is protected by
* copyright law, and remains the property of nymea GmbH. All rights, including
* reproduction, publication, editing and translation, are reserved. The use of
* this project is subject to the terms of a license agreement to be concluded
* with nymea GmbH in accordance with the terms of use of nymea GmbH, available
* under https://nymea.io/license
*
* GNU Lesser General Public License Usage
* Alternatively, this project may be redistributed and/or modified under the
* terms of the GNU Lesser General Public License as published by the Free
* Software Foundation; version 3. This project is distributed in the hope that
* it will be useful, but WITHOUT ANY WARRANTY; without even the implied
* warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with this project. If not, see <https://www.gnu.org/licenses/>.
*
* For any further details and any questions please contact us under
* contact@nymea.io or see our FAQ/Licensing Information on
* https://nymea.io/license/faq
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "scoreentry.h"

ScoreEntry::ScoreEntry() : TimeFrame()
{

}

double ScoreEntry::value() const
{
    return m_value;
}

void ScoreEntry::setValue(double value)
{
    m_value = value;
}

float ScoreEntry::weighting() const
{
    return m_weighting;
}

void ScoreEntry::setWeighting(float weighting)
{
    m_weighting = weighting;
}

bool ScoreEntry::isNull() const
{
    return m_startDateTime.isNull() && m_endDateTime.isNull() && m_value == 0 && m_weighting == 0;
}

bool ScoreEntry::operator==(const ScoreEntry &other) const
{
    return m_startDateTime == other.startDateTime() &&
            m_endDateTime == other.endDateTime() &&
            m_value == other.value() &&
            m_weighting == other.weighting();
}

bool ScoreEntry::operator!=(const ScoreEntry &other) const
{
    return !operator==(other);
}

qhash_result_t qHash(const ScoreEntry &scoreEntry, qhash_result_t seed)
{
    QtPrivate::QHashCombine hash;
    seed = hash(seed, scoreEntry.startDateTime());
    seed = hash(seed, scoreEntry.endDateTime());
    seed = hash(seed, scoreEntry.value());
    seed = hash(seed, scoreEntry.weighting());
    return seed;
}


ScoreEntries::ScoreEntries(const QList<ScoreEntry> &other) :
    QList<ScoreEntry>(other)
{

}

QVariant ScoreEntries::get(int index) const
{
    return QVariant::fromValue(at(index));
}

void ScoreEntries::put(const QVariant &variant)
{
    append(variant.value<ScoreEntry>());
}

int ScoreEntries::availableFutureSchedules(const QDateTime &currentDateTime) const
{
    int availableScheduleCount = 0;
    foreach (const ScoreEntry &entry, *this) {
        if (entry.endDateTime() > currentDateTime)
            availableScheduleCount++;
    }

    return availableScheduleCount;
}

void ScoreEntries::sortByWeighting()
{
    std::sort(begin(), end(), [](const ScoreEntry &a, const ScoreEntry &b) -> bool {
        return a.weighting() > b.weighting();
    });
}

void ScoreEntries::sortByStartDateTime()
{
    std::sort(begin(), end(), [](const ScoreEntry &a, const ScoreEntry &b) -> bool {
        return a.startDateTime() < b.startDateTime();
    });
}

ScoreEntry ScoreEntries::getScoreEntry(const QDateTime &currentDateTime) const
{
    foreach (const ScoreEntry &entry, *this) {
        if (currentDateTime >= entry.startDateTime() && currentDateTime < entry.endDateTime())
            return entry;
    }

    return ScoreEntry();
}

bool ScoreEntries::availableUpTo(const QDateTime &dateTime) const
{
    foreach (const ScoreEntry &entry, *this) {
        // Note: do not use TimeFrame::isActive here, since we include the endDateTime for the evaluation
        if (dateTime >= entry.startDateTime() && dateTime <= entry.endDateTime())
            return true;
    }

    return false;
}

QDebug operator<<(QDebug debug, const ScoreEntry &dataEntry)
{
    const QDebugStateSaver saver(debug);
    debug.nospace() << "ScoreEntry(" << dataEntry.startDateTime().toString("dd.MM.yyyy hh:mm");
    debug.nospace() << " - " << dataEntry.endDateTime().toString("dd.MM.yyyy hh:mm");
    debug.nospace() << ", " << dataEntry.weighting() << " | " << dataEntry.value() << ")";
    return debug;
}

QDebug operator<<(QDebug debug, const ScoreEntries &scoreEntries)
{
    const QDebugStateSaver saver(debug);
    debug.nospace() << "ScoreEntries(" << scoreEntries.count() << ")\n";
    for (int i = 0; i < scoreEntries.count(); i++) {
        debug.nospace() << " - " << i << " -> " << scoreEntries.at(i) << "\n";
    }
    return debug;
}
