/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright 2013 - 2022, nymea GmbH
* Contact: contact@nymea.io
*
* This file is part of nymea.
* This project including source code and documentation is protected by
* copyright law, and remains the property of nymea GmbH. All rights, including
* reproduction, publication, editing and translation, are reserved. The use of
* this project is subject to the terms of a license agreement to be concluded
* with nymea GmbH in accordance with the terms of use of nymea GmbH, available
* under https://nymea.io/license
*
* GNU Lesser General Public License Usage
* Alternatively, this project may be redistributed and/or modified under the
* terms of the GNU Lesser General Public License as published by the Free
* Software Foundation; version 3. This project is distributed in the hope that
* it will be useful, but WITHOUT ANY WARRANTY; without even the implied
* warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with this project. If not, see <https://www.gnu.org/licenses/>.
*
* For any further details and any questions please contact us under
* contact@nymea.io or see our FAQ/Licensing Information on
* https://nymea.io/license/faq
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "rootmeter.h"

#include "integrations/thing.h"

#include <QLoggingCategory>
Q_DECLARE_LOGGING_CATEGORY(dcNymeaEnergy)

RootMeter::RootMeter(Thing *parent)
    : QObject{parent},
      m_thing(parent)
{
    connect(m_thing, &Thing::stateValueChanged, this, [=](const StateTypeId &stateTypeId, const QVariant &value){
        if (m_thing->thingClass().getStateType(stateTypeId).name() == "currentPower") {
            emit currentPowerChanged(value.toDouble());
        }
    });
}

double RootMeter::currentPower() const
{
    return m_thing->stateValue("currentPower").toDouble();
}

double RootMeter::currentPowerPhaseA() const
{
    return m_thing->stateValue("currentPowerPhaseA").toDouble();
}

double RootMeter::currentPowerPhaseB() const
{
    return m_thing->stateValue("currentPowerPhaseB").toDouble();
}

double RootMeter::currentPowerPhaseC() const
{
    return m_thing->stateValue("currentPowerPhaseC").toDouble();
}

double RootMeter::currentPhaseA() const
{
    return m_thing->stateValue("currentPhaseA").toDouble();
}

double RootMeter::currentPhaseB() const
{
    return m_thing->stateValue("currentPhaseB").toDouble();
}

double RootMeter::currentPhaseC() const
{
    return m_thing->stateValue("currentPhaseC").toDouble();
}

double RootMeter::calculateAllowanceAmpere(Electricity::Phases phases, uint phasePowerLimit)
{
    double allowance = phasePowerLimit;
    double voltage = 230;

    if (phases.testFlag(Electricity::PhaseA) || phases == Electricity::PhaseNone) {
        if (m_thing->hasState("currentPhaseA")) {
            allowance = qMin(allowance, 1.0 * phasePowerLimit - m_thing->stateValue("currentPhaseA").toDouble());
        } else if (m_thing->hasState("currentPowerPhaseA")) {
            if (m_thing->hasState("voltagePhaseA"))
                voltage = m_thing->stateValue("voltagePhaseA").toDouble();

            allowance = qMin(allowance, 1.0 * phasePowerLimit - (m_thing->stateValue("currentPowerPhaseA").toDouble() / voltage));
        } else {
            // No currentPhaseA state... Assume single phase and deduct from currentPower
            double current = m_thing->stateValue("currentPower").toDouble() / 230;
            allowance = qMin(allowance, 1.0 * phasePowerLimit - current);
        }
    }
    if (phases.testFlag(Electricity::PhaseB) || phases == Electricity::PhaseNone) {
        if (m_thing->hasState("currentPhaseB")) {
            allowance = qMin(allowance, 1.0 * phasePowerLimit - m_thing->stateValue("currentPowerPhaseB").toDouble() / 230);
        } else if (m_thing->hasState("currentPowerPhaseB")) {
            if (m_thing->hasState("voltagePhaseB"))
                voltage = m_thing->stateValue("voltagePhaseB").toDouble();

            allowance = qMin(allowance, 1.0 * phasePowerLimit - (m_thing->stateValue("currentPowerPhaseB").toDouble() / voltage));
        } else {
            qCDebug(dcNymeaEnergy()) << "Asked to calculate allowence for Phase B on a single phase root meter.";
        }
    }

    if (phases.testFlag(Electricity::PhaseC) || phases == Electricity::PhaseNone) {
        if (m_thing->hasState("currentPhaseC")) {
            allowance = qMin(allowance, 1.0 * phasePowerLimit - m_thing->stateValue("currentPowerPhaseC").toDouble() / 230);
        } else if (m_thing->hasState("currentPowerPhaseC")) {
            if (m_thing->hasState("voltagePhaseC"))
                voltage = m_thing->stateValue("voltagePhaseC").toDouble();

            allowance = qMin(allowance, 1.0 * phasePowerLimit - (m_thing->stateValue("currentPowerPhaseC").toDouble() / voltage));
        } else {
            qCDebug(dcNymeaEnergy()) << "Asked to calculate allowence for Phase C on a single phase root meter.";
        }
    }

    return allowance;
}
