/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright 2013 - 2024, nymea GmbH
* Contact: contact@nymea.io
*
* This file is part of nymea.
* This project including source code and documentation is protected by
* copyright law, and remains the property of nymea GmbH. All rights, including
* reproduction, publication, editing and translation, are reserved. The use of
* this project is subject to the terms of a license agreement to be concluded
* with nymea GmbH in accordance with the terms of use of nymea GmbH, available
* under https://nymea.io/license
*
* GNU Lesser General Public License Usage
* Alternatively, this project may be redistributed and/or modified under the
* terms of the GNU Lesser General Public License as published by the Free
* Software Foundation; version 3. This project is distributed in the hope that
* it will be useful, but WITHOUT ANY WARRANTY; without even the implied
* warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with this project. If not, see <https://www.gnu.org/licenses/>.
*
* For any further details and any questions please contact us under
* contact@nymea.io or see our FAQ/Licensing Information on
* https://nymea.io/license/faq
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "mailclient.h"

#include <QHttpMultiPart>
#include <QFile>
#include <QFileInfo>
#include <QNetworkReply>
#include <QUrlQuery>

#include <QLoggingCategory>
Q_DECLARE_LOGGING_CATEGORY(dcChargingSessions)

MailClient::MailClient(QObject *parent)
    : QObject{parent},
    m_networkAccessManager{new QNetworkAccessManager(this)}
{

}

QNetworkReply *MailClient::sendEmail(const QString &senderName, const QString &senderEmail, const QStringList recipientEmails, const QString &subject, const QString &body, const QStringList attachments)
{
    qCDebug(dcChargingSessions()) << "Sending reports as" << senderName << QString("<%1>").arg(senderEmail) << "to" << recipientEmails << "subject:" << subject;;

    QHttpMultiPart *multiPart = new QHttpMultiPart(QHttpMultiPart::FormDataType);

    QHttpPart fromPart;
    fromPart.setHeader(QNetworkRequest::ContentDispositionHeader, "form-data; name=\"from\"");
    fromPart.setBody(senderName.toUtf8() + "<" + senderEmail.toUtf8() + ">");
    multiPart->append(fromPart);

    foreach (const QString &recipient, recipientEmails) {
        QHttpPart toPart;
        toPart.setHeader(QNetworkRequest::ContentDispositionHeader, "form-data; name=\"to\"");
        toPart.setBody(recipient.toUtf8());
        multiPart->append(toPart);
    }

    QHttpPart subjectPart;
    subjectPart.setHeader(QNetworkRequest::ContentDispositionHeader, "form-data; name=\"subject\"");
    subjectPart.setBody(subject.toUtf8());
    multiPart->append(subjectPart);

    QHttpPart textPart;
    textPart.setHeader(QNetworkRequest::ContentDispositionHeader, "form-data; name=\"text\"");
    textPart.setBody(body.toUtf8());
    multiPart->append(textPart);

    QHttpPart replyPart;
    replyPart.setHeader(QNetworkRequest::ContentDispositionHeader, "form-data; name=\"h:reply-To\"");
    replyPart.setBody(senderEmail.toUtf8());
    multiPart->append(replyPart);

    foreach (const QString &attachment, attachments) {
        QFile *file = new QFile(QString(attachment).remove("file://"));
        file->setParent(multiPart);
        if (file->open(QIODevice::ReadOnly)) {
            QFileInfo attachmentFileInfo(attachment);
            QString headerLine = QString("form-data; name=\"attachment\"; filename=\"%1\"; contentType=\"text/csv\"").arg(attachmentFileInfo.fileName());
            qCDebug(dcChargingSessions()) << "Attache file" << attachment << headerLine;

            QHttpPart imagePart;
            imagePart.setHeader(QNetworkRequest::ContentDispositionHeader, QVariant(headerLine));
            imagePart.setBodyDevice(file);
            multiPart->append(imagePart);
        }
    }

    QUrl url("https://api.eu.mailgun.net/v3/sendmail.nymea.io/messages");

    QNetworkRequest request(url);
    QByteArray credentials("api:key-6da6a80f4d965d0c168114d1d665a875");
    request.setRawHeader("Authorization", "Basic " + credentials.toBase64());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "multipart/form-data; boundary=" + multiPart->boundary());

    QNetworkReply *reply = m_networkAccessManager->post(request, multiPart);
    multiPart->setParent(reply);

    return reply;
}
