// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-chargingsessions.
*
* nymea-energy-plugin-chargingsessions is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-chargingsessions is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-chargingsessions. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "testchargingsessions.h"

ChargingSessionsTests::ChargingSessionsTests(QObject *parent)
    : ChargingSessionsTestBase{parent}
{

}

void ChargingSessionsTests::getConfiguration()
{
    QVariant response = injectAndWait("ChargingSessions.GetConfiguration");
    qCDebug(dcTests()) << qUtf8Printable(QJsonDocument::fromVariant(response).toJson(QJsonDocument::Indented));

    QVERIFY(response.toMap().value("status").toString() == "success");

    QVERIFY(response.toMap().value("params").toMap().contains("configuration"));
    QVariantMap configurationMap = response.toMap().value("params").toMap().value("configuration").toMap();
    QVERIFY(configurationMap.contains("reporterName"));
    QVERIFY(configurationMap.contains("reporterEmail"));
    QVERIFY(configurationMap.contains("recipientEmails"));
}

void ChargingSessionsTests::setConfiguration_data()
{
    QTest::addColumn<QString>("reporterName");
    QTest::addColumn<QString>("reporterEmail");
    QTest::addColumn<QStringList>("recipientEmails");
    QTest::addColumn<QString>("expectedError");

    QTest::newRow("Default") << "Chuck Norris" << "hello@world.com" << QStringList({"chuck@norris.org", "chuck@norris.com"}) << "ChargingSessionsErrorNoError";
    QTest::newRow("Invalid email recipient") << "Chuck Norris" << "hello@world.com" << QStringList({"chuck-norris.org"}) << "ChargingSessionsErrorInvalidEmail";
    QTest::newRow("Invalid reporter email") << "Chuck Norris" << "hello@world" << QStringList({"chuck@norris.org", "chuck@norris.com"}) << "ChargingSessionsErrorInvalidEmail";
    QTest::newRow("New top level domain") << "Chuck Norris" << "hello@world.com" << QStringList({"chuck.norris@nymea.energy"}) << "ChargingSessionsErrorNoError";
    QTest::newRow("New top level domain") << "Chuck Norris" << "chuck.norris@nymea.energy" << QStringList({"chuck.norris@nymea.energy"}) << "ChargingSessionsErrorNoError";
}

void ChargingSessionsTests::setConfiguration()
{
    QFETCH(QString, reporterName);
    QFETCH(QString, reporterEmail);
    QFETCH(QStringList, recipientEmails);
    QFETCH(QString, expectedError);

    QVariant response, notification;
    QSignalSpy notificationSpy(m_mockTcpServer, &MockTcpServer::outgoingData);

    QVariantMap configurationMap;
    configurationMap.insert("reporterName", reporterName);
    configurationMap.insert("reporterEmail", reporterEmail);
    configurationMap.insert("recipientEmails", recipientEmails);

    QVariantMap params;
    params.insert("configuration", configurationMap);

    // Initially disable
    response = injectAndWait("ChargingSessions.SetConfiguration", params);
    qCDebug(dcTests()) << qUtf8Printable(QJsonDocument::fromVariant(response).toJson(QJsonDocument::Indented));

    QVariantMap responseParams = response.toMap().value("params").toMap();
    QCOMPARE(responseParams.value("chargingSessionsError").toString(), expectedError);

    if (expectedError == "ChargingSessionsErrorNoError") {
        // Verify the ConfigurationChanged notification has been emitted
        if (notificationSpy.count() == 0) notificationSpy.wait();
        notification = checkNotification(notificationSpy, "ChargingSessions.ConfigurationChanged");
        //qCDebug(dcTests()) << qUtf8Printable(QJsonDocument::fromVariant(notification).toJson());
        QVERIFY(notificationSpy.count() >= 1);

        QVariantMap notificationConfiguration = notification.toMap().value("params").toMap().value("configuration").toMap();
        QCOMPARE(notificationConfiguration.value("reporterName").toString(), reporterName);
        QCOMPARE(notificationConfiguration.value("reporterEmail").toString(), reporterEmail);
        QCOMPARE(notificationConfiguration.value("recipientEmails").toStringList(), recipientEmails);

        // Get the configuration and make sure the values are the same
        response = injectAndWait("ChargingSessions.GetConfiguration");
        //qCDebug(dcTests()) << qUtf8Printable(QJsonDocument::fromVariant(response).toJson(QJsonDocument::Indented));
        QVariantMap responseConfigurationMap = response.toMap().value("params").toMap().value("configuration").toMap();
        QCOMPARE(responseConfigurationMap.value("reporterName").toString(), reporterName);
        QCOMPARE(responseConfigurationMap.value("reporterEmail").toString(), reporterEmail);
        QCOMPARE(responseConfigurationMap.value("recipientEmails").toStringList(), recipientEmails);
    }
}


QTEST_MAIN(ChargingSessionsTests)
