/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright 2013 - 2020, nymea GmbH
* Contact: contact@nymea.io
*
* This file is part of maveod.
* This project including source code and documentation is protected by
* copyright law, and remains the property of nymea GmbH. All rights, including
* reproduction, publication, editing and translation, are reserved. The use of
* this project is subject to the terms of a license agreement to be concluded
* with nymea GmbH in accordance with the terms of use of nymea GmbH, available
* under https://nymea.io/license
*
* For any further details and any questions please contact us under
* contact@nymea.io or see our FAQ/Licensing Information on
* https://nymea.io/license/faq
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef LEDCONTROLLER_H
#define LEDCONTROLLER_H

#include <QColor>
#include <QObject>
#include <QFile>

class LedController : public QObject
{
    Q_OBJECT
public:
    enum LedStatus {
        LedStatusSuccess = 0x00,
        LedStatusFailure = 0xff
    };
    Q_ENUM(LedStatus)

    enum LedCommand {
        LedCommandSetColor          = 0x00,
        LedCommandGlowColor         = 0x01,
        LedCommandBlink             = 0x02,
        LedCommandFlash             = 0x03,
        LedCommandDoubleFlash       = 0x04,
        LedCommandSnooze            = 0x05,
        LedCommandCycleRgb          = 0x06,
        LedCommandCycleRainbow      = 0x07,
        LedCommandCycleRedGreen     = 0x08,
        LedCommandCycleBlueRed      = 0x09,
        LedCommandCycleGreenBlue    = 0x0a,
        LedCommandCycleRedWhite     = 0x0b,
        LedCommandCycleGreenWhite   = 0x0c,
        LedCommandCycleBlueWhite    = 0x0d,
        LedCommandSoftwareReset     = 0xff
    };
    Q_ENUM(LedCommand)

    enum LedAnimation {
        LedAnimationNone            = 0x00,
        LedAnimationLinear          = 0x01,
        LedAnimationQuadraticIn     = 0x02,
        LedAnimationQuadraticOut    = 0x03,
        LedAnimationQuadraticInOut  = 0x04,
        LedAnimationCubicIn         = 0x05,
        LedAnimationCubicOut        = 0x06,
        LedAnimationCubicInOut      = 0x07,
        LedAnimationLogistic        = 0x08,
    };
    Q_ENUM(LedAnimation)

    explicit LedController(const QString &i2cBusFileName = "/dev/i2c-1", quint8 i2cAddress = 0x08, QObject *parent = nullptr);

    QString i2cBusFileName() const;
    quint8 i2cAddress() const;

    // Data from the controller
    LedCommand command() const;
    LedAnimation animation() const;
    quint16 duration() const;
    QColor color() const;
    QString controllerVersion() const;

    bool setLed(LedCommand command, LedAnimation animation, quint16 duration, const QColor &color);

private:
    QFile m_i2cBusFile;
    QString m_i2cBusFileName;
    quint8 m_i2cAddress;
    QString m_controllerVersion;

    LedCommand m_command = LedCommandSetColor;
    LedAnimation m_animation = LedAnimationNone;
    quint16 m_duration = 1000;
    QColor m_color = QColor(Qt::white);

    bool readData();

    void setCommand(LedCommand command);
    void setAnimation(LedAnimation animation);
    void setDuration(quint16 duration);
    void setColor(const QColor &color);

signals:
    void commandChanged(LedCommand command);
    void animationChanged(LedAnimation animation);
    void durationChanged(quint16 duration);
    void colorChanged(const QColor &color);

};

#endif // LEDCONTROLLER_H
